/*
 *  sudo version 1.1 allows users to execute commands as root
 *  Copyright (C) 1991  The Root Group, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  If you make modifications to the source, we would be happy to have
 *  them to include in future releases.  Feel free to send them to:
 *      Jeff Nieusma                       nieusma@rootgroup.com
 *      3959 Arbol CT                      (303) 447-8093
 *      Boulder, CO 80301-1752             
 *
********************************************************************************
* parse.c, sudo project
* David R. Hieb
* March 18, 1991
*
* routines to implement and maintain the parsing and list management.
*******************************************************************************/
#include <stdio.h>
#include <strings.h>
#include <ctype.h>
#include <afs/param.h>
#include <afs/stds.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <rx/rx.h>
#include <rx/xdr.h>
#include <rx/rxkad.h>
#include <afs/auth.h>
#include <afs/cellconfig.h>



#include "sudo.h"

/* there are 5 main lists (User, UNIXGroup, PtsGroup, Host_Alias, Cmnd_Alias) 
   and 1 extra list */


extern char *sudo_user, *sudo_host, *sudo_cmnd;
extern FILE *yyin, *yyout;

int user_list_found = FALSE;
int unixgroup_list_found = FALSE;
int ptsgroup_list_found = FALSE;
int list_num, new_list[NUM_LISTS];
int parse_error = FALSE, found_user = FALSE;
int next_type, num_host_alias = 0, num_cmnd_alias = 0;
int num_unixgroup = 0, num_ptsgroup = 0;
char entity_permitted[80], entity_depermitted[80];
LINK tmp_ptr, reset_ptr, save_ptr, list_ptr[NUM_LISTS];

/*******************************************************************************
* inserts a node into list 'list_num' and updates list_ptr[list_num]
*******************************************************************************/
void insert_member(list_num, token_type, op_type, data_string)
int token_type, list_num;
char op_type;
char *data_string;
{
    LINK temp_ptr;

    temp_ptr = (LINK)malloc(sizeof(LIST));
    temp_ptr->type = token_type;
    temp_ptr->op = op_type;
    temp_ptr->data = (char *)malloc(strlen(data_string)+1);
    strcpy(temp_ptr->data, data_string);
    temp_ptr->next = (new_list[list_num] == TRUE) ? NULL : list_ptr[list_num];

    list_ptr[list_num] = temp_ptr;
}

/*******************************************************************************
* diagnostic list printing utility that prints list 'list_num'
*******************************************************************************/
void print_list(list_num)
int list_num;
{
LINK tmptmp_ptr;

tmptmp_ptr = list_ptr[list_num];

while (list_ptr[list_num] != NULL) {
    printf("type = %d, op = %c, data = %s\n", list_ptr[list_num]->type,
        list_ptr[list_num]->op, list_ptr[list_num]->data);
    tmp_ptr = list_ptr[list_num];
    list_ptr[list_num] = tmp_ptr->next;
    }
list_ptr[list_num] = tmptmp_ptr;
}

/*******************************************************************************
* delete list utility that deletes list 'list_num'
*******************************************************************************/
void delete_list(list_num)
int list_num;
{
while (list_ptr[list_num] != NULL) {
    tmp_ptr = list_ptr[list_num];
    list_ptr[list_num] = tmp_ptr->next;
    free(tmp_ptr);
    }
}

/*******************************************************************************
* this routine is what the lex/yacc code calls to build the different lists.
* once the lists are all built, control eventually returns to validate().
*******************************************************************************/
int call_back(token_type, op_type, data_string)
int token_type;
char op_type;
char *data_string;
{
   /* all nodes start out in the extra list since 
      the node name is received last */
   list_num = EXTRA_LIST;

   /*
    * if the last received node is TYPE1, then we can classify the list
    * and effectively transfer the extra list to the correct list type.
    */
   if (token_type == TYPE1) {
      /* we have just build a "Host_Alias" list */
      if (strcmp(data_string, "Host_Alias") == 0) {
	 list_num = HOST_LIST;
	 if (num_host_alias > 0) {
            reset_ptr->next = list_ptr[HOST_LIST];
	 }
	 num_host_alias++;
      }
      /* we have just build a "Cmnd_Alias" list */
      else if (strcmp(data_string, "Cmnd_Alias") == 0) {
	 list_num = CMND_LIST;
	 if (num_cmnd_alias > 0) {
            reset_ptr->next = list_ptr[CMND_LIST];
            }
	 num_cmnd_alias++;
      }
      else if ( data_string[0] == ':' ) { 
	 /* we have a UNIX group here */ 
	 list_num = UNIXGROUP_LIST;
	 if ( num_unixgroup > 0 ) { 
	    reset_ptr->next = list_ptr[UNIXGROUP_LIST];
	 }
	 num_unixgroup++;
      }
      else if ( data_string[0] == '~' ) { 
	 /* we have an AFS  pts Group here */ 
	 list_num = PTSGROUP_LIST;
	 if ( num_ptsgroup > 0 ) { 
	    reset_ptr->next = list_ptr[PTSGROUP_LIST];
	 }
	 num_ptsgroup++;
      }
      /* we have just build a "User" list */
      else {
         /* Remember only the most recent occurence of USER. */
         if( strcmp( data_string, sudo_user ) == 0 ) { 
	    list_num = USER_LIST;
            user_list_found = TRUE;
            found_user = TRUE;
            delete_list( list_num );
         }
         else { 
         /* This user is different from sudo_user */
            new_list[EXTRA_LIST] = TRUE;
            delete_list(EXTRA_LIST);
            return(NOT_FOUND_USER);
         }
      }
      new_list[EXTRA_LIST] = TRUE;
      new_list[list_num] = FALSE;
      list_ptr[list_num] = list_ptr[EXTRA_LIST];
   }
   
   /* actually link the new node into list 'list_num' */
   insert_member(list_num, token_type, op_type, data_string);


   if (new_list[list_num] == TRUE) {
      reset_ptr = list_ptr[list_num];
      new_list[list_num] = FALSE;
   }
   
   /*
    * we process one user record at a time from the sudoers file. if we
    * find the user were looking for, we return to lex/yacc declaring 
    * that we have done so. otherwise, we reset the user list, delete the 
    * nodes and start over again looking for the user.
    */
   if (user_list_found == TRUE) {
      if (list_ptr[list_num]->type == TYPE1 &&
	 strcmp(list_ptr[list_num]->data, sudo_user) == 0) {
         user_list_found = FALSE;
	 return(FOUND_USER);
      }
      else {
	 new_list[list_num] = TRUE;
	 user_list_found = FALSE;
	 delete_list(list_num);
      }
   }
   return(NOT_FOUND_USER);
}

/*******************************************************************************
* this routine is called from pts_cmnd_check() to resolve whether or not
* a user is permitted to perform a to-yet-be-determined command for
* a certain host name.
*******************************************************************************/
int pts_host_type_ok( search_ptr )
LINK search_ptr;
{
  
   int  retval;
   LINK my_hostlist;

   my_hostlist = list_ptr[HOST_LIST];
#ifdef DEBUG
   printf("Checking for host type : %s\n", search_ptr->data );
#endif
   /* check for the reserved keyword 'ALL'. if so, don't check the host name */
   if (strcmp(search_ptr->data, "ALL") == 0) {
      return(TRUE);
   }
   /* this case is the normal lowercase hostname */
   else if (isupper(search_ptr->data[0]) == FALSE) {
#ifdef DEBUG
      printf("We are on host %s ? .....", search_ptr->data );
#endif
      retval = strcmp(search_ptr->data, sudo_host);
      if( retval == 0 ) { 
         if( search_ptr->op == '!' ) { 
#ifdef DEBUG
            printf("Yes...Negative Permission Found.\n");
#endif
            return( FALSE ); 
         }
         else { 
#ifdef DEBUG 
            printf("Yes..Affirmative Permission for host found.\n");
#endif 
            return( TRUE );
         }
      }
      else { 
#ifdef DEBUG
         printf("No.. \n");
#endif 
         return( KEEP_TRYING );
      }  
   }
   /* by now we have a Host_Alias that will have to be expanded */
   else {
#ifdef DEBUG
      printf("Expanding Host Alias %s\n", search_ptr->data );
#endif
      save_ptr = my_hostlist;
      while (my_hostlist != NULL) {
	 if ((my_hostlist->type == TYPE2) &&
	     (strcmp(my_hostlist->data, 
		     search_ptr->data) == 0)) {
            next_type = my_hostlist->next->type;
            tmp_ptr = my_hostlist;
            my_hostlist = tmp_ptr->next;
            while (next_type == TYPE3) {
#ifdef DEBUG
                printf("Checking if we are on host %s\n", 
                       my_hostlist->data );                
#endif 
                retval = pts_host_type_ok( my_hostlist );
                if( retval == TRUE ) { 
                   return( TRUE );
                }
                if( retval == FALSE ) { 
                   return( FALSE );
                }
/*
                if (strcmp(my_hostlist->data, sudo_host) == 0) {
                    my_hostlist = save_ptr;
                    return(TRUE);
                    }
*/
                if (my_hostlist->next != NULL) {
                    next_type = my_hostlist->next->type;
                    tmp_ptr = my_hostlist;
                    my_hostlist = tmp_ptr->next;
                    }
                else {
                    next_type = ~TYPE3;
                    }
                }
            }
        else {
            tmp_ptr = my_hostlist;
            my_hostlist = tmp_ptr->next;
            }
        }
    return(KEEP_TRYING);
    }
}


/*******************************************************************************
* this routine is called from pts_cmnd_check() to resolve whether or not
* a user is permitted to perform a certain command on the already
* established host.
*******************************************************************************/
int pts_cmnd_type_ok(search_ptr)
LINK search_ptr;
{

   LINK  save_ptr, tmp_ptr; 
   int   retval;


   /* check for the reserved keyword 'ALL'. */
   if (strcmp(search_ptr->data, "ALL") == 0) {
      /* if the command has an absolute path, let them do it */
      if (sudo_cmnd[0] == '/') {
	 return(MATCH);
      }
      /* if the command does not have an absolute path, forget it */
      else {
	 return(NO_MATCH);
      }
   }
   /* if the command has an absolute path, check it out */
   else if (search_ptr->data[0] == '/') {

      /* op  |   data   | return value
       *---------------------------------
       * ' ' | No Match | return(NO_MATCH) 
       * '!' | No Match | return(NO_MATCH) 
       * ' ' |  A Match | return(MATCH) 
       * '!' |  A Match | return(QUIT_NOW)
       *
       * these special cases are important in subtracting from the
       * Universe of commands in something like:
       *   user machine=ALL,!/bin/rm,!/etc/named ...
       */
      if (strcmp(search_ptr->data, sudo_cmnd) == 0) {
	 if (search_ptr->op == '!') {
	    return(QUIT_NOW);
	 }
	 else {
            return(MATCH);
	 }
      }
      else {
	 return(NO_MATCH);
      }
   }
   /* by now we have a Cmnd_Alias that will have to be expanded */
   else {
      save_ptr = list_ptr[CMND_LIST];
      tmp_ptr  = list_ptr[CMND_LIST]; 
      while (tmp_ptr != NULL) {
	 if ((tmp_ptr->type == TYPE2) &&
	     (strcmp(tmp_ptr->data, 
		     search_ptr->data) == 0)) {
            next_type = tmp_ptr->next->type;
            tmp_ptr = tmp_ptr->next; 
            while (next_type == TYPE3) {
                retval = pts_cmnd_type_ok( tmp_ptr );
                if( retval == MATCH ) return(MATCH);
                if( retval == QUIT_NOW ) return(QUIT_NOW);
                if (tmp_ptr->next != NULL) {
		   next_type = tmp_ptr->next->type;
		   tmp_ptr = tmp_ptr->next;
		}
                else {
		   next_type = ~TYPE3;
		}
	     }
	 }
	 else {
            tmp_ptr = tmp_ptr->next;
	 }
      }
      return(NO_MATCH);
   }
}

/*******************************************************************************
* this routine is called from validate() after the call_back() routine
* has built all the possible lists. this routine steps thru the user list
* calling on host_type_ok() and cmnd_type_ok() trying to resolve whether
* or not the user will be able to execute the command on the host.
*******************************************************************************/
int cmnd_check()
{
   int return_code;

   while (list_ptr[USER_LIST] != NULL) {
      if ((list_ptr[USER_LIST]->type == TYPE2) && 
          (pts_host_type_ok(list_ptr[USER_LIST]) == TRUE) ) {
	 next_type = list_ptr[USER_LIST]->next->type;
	 tmp_ptr = list_ptr[USER_LIST]; 
	 list_ptr[USER_LIST] = tmp_ptr->next;
	 while (next_type == TYPE3) {
            return_code = pts_cmnd_type_ok(list_ptr[USER_LIST]);
            if (return_code == MATCH) {
               strcpy( entity_permitted, sudo_user );
	       return(VALIDATE_OK);
	    }
            else if (return_code == QUIT_NOW) {
               strcpy( entity_depermitted, sudo_user );
	       return(VALIDATE_NOT_OK);
	    }
            if (list_ptr[USER_LIST]->next != NULL) {
	       next_type = list_ptr[USER_LIST]->next->type;
	       tmp_ptr = list_ptr[USER_LIST];
	       list_ptr[USER_LIST] = tmp_ptr->next;
	    }
            else {
	       next_type = ~TYPE3;
	    }
	 }
      }
      else {
	 tmp_ptr = list_ptr[USER_LIST];
	 list_ptr[USER_LIST] = tmp_ptr->next;
      }
   }
   strcpy( entity_depermitted, sudo_user );
   return(VALIDATE_NO_PERMISSION);
}
 
/*******************************************************************************
* this routine sis called from pts_check() if the user is a member of the 
* current pts group. It checks to see if the group has been permitted to use
* cmnd, and returns status 
*******************************************************************************/

#ifdef PTSGRP_CHECK

int pts_check_cmnd( lst_ptr )
LINK lst_ptr;
{

   LINK search_ptr; 
   int return_code;
   char *this_entity;

   search_ptr = lst_ptr->next; 
   this_entity = lst_ptr->data;
   /* We want to search only ONE Pts group's permission. */ 
   while( (search_ptr != (LINK)NULL) && 
	 (search_ptr->type != TYPE1) ) { 
      if ((search_ptr->type == TYPE2) && 
	  (pts_host_type_ok(search_ptr) == TRUE) ) {
	 next_type = search_ptr->next->type;
	 tmp_ptr = search_ptr;
	 search_ptr = tmp_ptr->next;
	 while (next_type == TYPE3) {
#ifdef DEBUG
            printf("Checking for cmnd type: %s\n", search_ptr->data );
#endif DEBUG
            return_code = pts_cmnd_type_ok(search_ptr);
            if (return_code == MATCH) {
	       return(VALIDATE_OK);
	    }
            else if (return_code == QUIT_NOW) {
	       return(VALIDATE_NOT_OK);
	    }
            if (search_ptr->next != NULL) {
	       next_type = search_ptr->next->type;
	       tmp_ptr = search_ptr;
	       search_ptr = tmp_ptr->next;
	    }
            else {
	       next_type = ~TYPE3;
	    }
	 }
      }
      else {
	 tmp_ptr = search_ptr;
	 search_ptr = tmp_ptr->next;
        
      }
   }
   return(VALIDATE_NO_PERMISSION);
}



/*
 * pts_check()
 * 
 * This function checks to see if the user is a member of any of the AFS 
 * pts groups parsed in by the program.
 */

int pts_check()
{

   long is_member_flag;
   long pts_retval, retval;
   int pts_member_found;
   int security_level;
   char *ConfDir, *CellName;
   char *grp_name;
   LINK temp_ptr;

   /* First delete the USER_LIST. We may need to re-use it. */ 
   delete_list(USER_LIST);

   security_level = AFS_SECURITY_LEVEL;
   ConfDir        = AFS_CONFIG_DIRECTORY;
   CellName       = AFS_CELLNAME;

   pts_retval = pr_Initialize( security_level,
			      ConfDir,
			      CellName );
   if( pts_retval ) { 
      fprintf(stderr, "Error initializing AFS Protection DataBase Query.\n");
      fprintf(stderr, "Aborting Query.\n");
      return( VALIDATE_NO_USER );
   }
   temp_ptr = list_ptr[PTSGROUP_LIST];
   while( (num_ptsgroup > 0) && (temp_ptr  != (LINK)NULL) ) { 
      if( temp_ptr->type == TYPE1 ) { 

	 /* An AFS pts group name */ 

	 if( temp_ptr->data[0] != '~' ) { 
	    fprintf(stderr, "Inconsistency in Internal Lists Detected.\n");
	    fprintf(stderr, "Send Mail to repair@rpi.edu.\n");
	    return(VALIDATE_ERROR);
	 }
	 grp_name = &(temp_ptr->data[1]);
#ifdef DEBUG
         printf("Checking pts Group: %s\n", grp_name);
#endif DEBUG
	 pts_retval = pr_IsAMemberOf( sudo_user, grp_name, &is_member_flag );
	 if( pts_retval ) { 
	    fprintf(stderr, "Error Querying Protection Database.\n");
	    fprintf(stderr, "Aborting Query.\n");
	    return( VALIDATE_NO_USER );
	 }
	 if( is_member_flag ) { 

	    /* This person is indeed a member of this list. */ 

	    pts_member_found = TRUE;
            found_user = TRUE;
#ifdef DEBUG
            printf("Found in group %s, checking further\n", grp_name);
#endif DEBUG
	    retval = pts_check_cmnd( temp_ptr );
	    if( retval == VALIDATE_OK ) { 
               strcpy( entity_permitted, temp_ptr->data );
	       return VALIDATE_OK ;
	    }
            if( retval == VALIDATE_NOT_OK ) {
               strcpy( entity_depermitted, temp_ptr->data );
               return VALIDATE_NOT_OK;
            }
	 }
	 temp_ptr = temp_ptr->next;
      }	    
      else { 
	 temp_ptr = temp_ptr->next; 
      }
   }
   if( (pts_member_found == TRUE) ) {
      strcpy( entity_depermitted, "No Permission" );
      return( VALIDATE_NO_PERMISSION );
   }
   else {
      strcpy( entity_depermitted, "No Membership" );
      return( VALIDATE_NO_USER ); 
   }
}

#endif PTSGRP_CHECK

#ifdef UNIXGRP_CHECK


int unixgrp_check()
{

   long is_member_flag;
   int grps[MAX_UNIXGRPS];
   int  num_grps, retval, i;
   int unixgrp_member_found;
   char *grp_name;
   struct group *grp_ptr;
   LINK temp_ptr;

   /* First delete the USER_LIST. We may need to re-use it. */ 
   delete_list(USER_LIST);


   /* Get GID's of all groups the user is a member of. */

   num_grps = getgroups( MAX_UNIXGRPS, grps );

   temp_ptr = list_ptr[UNIXGROUP_LIST];
   while( (num_unixgroup > 0) && (temp_ptr  != (LINK)NULL) ) { 
      if( temp_ptr->type == TYPE1 ) { 

	 /* An UNIX group name */ 

	 if( temp_ptr->data[0] != ':' ) { 
	    fprintf(stderr, "Inconsistency in Internal Lists Detected.\n");
	    fprintf(stderr, "Send Mail to repair@rpi.edu.\n");
	    return(VALIDATE_ERROR);
	 }
	 grp_name = &(temp_ptr->data[1]);
#ifdef DEBUG
         printf("Checking Unix Group: %s\n", grp_name);
#endif DEBUG	 
	 
	 grp_ptr = (struct group *) getgrnam( grp_name );
	 if( grp_ptr ) { 

	    /* The group exists. */

	    for( i = 0; i < num_grps ; i ++ ) { 

	       if( grp_ptr->gr_gid == grps[i] ) { 

		  /* This person is indeed a member of this group. */ 

		  unixgrp_member_found = TRUE;
                  found_user = TRUE;
#ifdef DEBUG		  
		  printf("Found in group %s, checking further\n", grp_name);
#endif DEBUG		  
		  retval = pts_check_cmnd( temp_ptr );
		  if( retval == VALIDATE_OK ) { 
		     strcpy( entity_permitted, temp_ptr->data );
		     return VALIDATE_OK ;
		  }
		  if( retval == VALIDATE_NOT_OK ) {
		     strcpy( entity_depermitted, temp_ptr->data );
		     return VALIDATE_NOT_OK;
		  }
	       }
	    }
	 }
	 temp_ptr = temp_ptr->next;
      }
      else { 
	 temp_ptr = temp_ptr->next;
      }
   }

   if( (unixgrp_member_found == TRUE) ) {
      strcpy( entity_depermitted, "No Permission" );
      return( VALIDATE_NO_PERMISSION );
   }
   else {
      strcpy( entity_depermitted, "No Membership" );
      return( VALIDATE_NO_USER ); 
   }
}

#endif UNIXGRP_CHECK


/*******************************************************************************
* this routine is called from the sudo.c module and tries to validate
* the user, host and command triplet.
*******************************************************************************/
int validate()
{
   FILE *sudoers_fp;
   int i, return_code;

   if ((sudoers_fp = fopen(SUDOERS, "r")) == NULL ) {
      perror(SUDOERS);
      log_error( NO_SUDOERS_FILE );
      exit(1);
   }
   
   yyin = sudoers_fp;
   yyout = stdout;

   for (i = 0; i < NUM_LISTS; i++)
      new_list[i] = TRUE;
   reset_ptr = (LINK) NULL; 
   found_user = FALSE;
   
   /*
    * yyparse() returns with one of 3 values: 0) yyparse() worked fine; 
    * 1) yyparse() failed; FOUND_USER) the user was found and yyparse()
    * was returned from prematurely.
    */
   return_code = yyparse();

   /* don't need to keep this open... */
   (void) fclose (sudoers_fp);

   /* 
    * There's no point in running setuid more than necessary.
    *
    */

   setruid( sudo_uid ); 

#ifdef DEBUG 
   print_list(USER_LIST);
#endif
   /* if a parsing error occurred, set return_code accordingly */
   if (parse_error == TRUE) {
      return_code = PARSE_ERROR;
      return(VALIDATE_ERROR);
   }

   /* handle the 3 cases sequentially &*/
   /* First direct user match. */

   if( found_user == TRUE ) { 
      return_code = cmnd_check();
      if( (return_code == VALIDATE_OK) || (return_code == VALIDATE_NOT_OK) ) {
         if( (return_code == VALIDATE_NOT_OK) ) { 
            strcpy( entity_depermitted, sudo_user );
         }
	 delete_list(USER_LIST);
	 delete_list(HOST_LIST);
	 delete_list(CMND_LIST);
	 delete_list(UNIXGROUP_LIST);
	 delete_list(PTSGROUP_LIST);
	 return(return_code);
      }
   }

#ifdef PTSGRP_CHECK

   /* Check if the user is in a pts group. */
#ifdef DEBUG   
   printf("checking pts groups\n");
#endif
   return_code = pts_check();
   if( (return_code == VALIDATE_OK) || (return_code == VALIDATE_NOT_OK) ) {
      delete_list(HOST_LIST);
      delete_list(CMND_LIST);
      delete_list(UNIXGROUP_LIST);
      delete_list(PTSGROUP_LIST);
      return(return_code);
   }
#endif PTSGRP_CHECK

#ifdef UNIXGRP_CHECK
   
#ifdef DEBUG
   printf("checking Unix groups\n");
#endif
   
   return_code = unixgrp_check(); 
   if( (return_code == VALIDATE_OK) || (return_code == VALIDATE_NOT_OK) ) { 
      delete_list(HOST_LIST);
      delete_list(CMND_LIST);
      delete_list(UNIXGROUP_LIST);
      delete_list(PTSGROUP_LIST);
      return(return_code);
   }
#endif UNIXGRP_CHECK

   delete_list(HOST_LIST);
   delete_list(CMND_LIST);
   delete_list(UNIXGROUP_LIST);
   delete_list(PTSGROUP_LIST);
   if( found_user )
     return(VALIDATE_NO_PERMISSION);
   else
     return(return_code);
}

