/*
 *  sudo version 1.1 allows users to execute commands as root
 *  Copyright (C) 1991  The Root Group, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  If you make modifications to the source, we would be happy to have
 *  them to include in future releases.  Feel free to send them to:
 *      Jeff Nieusma                       nieusma@rootgroup.com
 *      3959 Arbol CT                      (303) 447-8093
 *      Boulder, CO 80301-1752             
 * 
 ****************************************************************
 *
 *  logging.c
 *
 *  this file supports the general logging facilities
 *  if you want to change any error messages, this is probably
 *  the place to be...
 *
 *  Jeff Nieusma   Thu Mar 21 23:39:04 MST 1991
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include "sudo.h"

void log_error();
void reapchild();
static void send_mail();
static void reapchild();
static int appropriate();

static char logline[MAXLOGLEN+8];
static char locallogline[MAXLOGLEN+8];
static char *logfile_name;
extern char entity_permitted[], entity_depermitted[];

/**********************************************************************
 *
 *  log_error()
 *
 *  This function attempts to deliver mail to ALERTMAIL and either
 *  syslogs the error or writes it to the log file
 */

void log_error( code )
int code;
{
   char cwd[MAXPATHLEN+1];
   char *line_delim, *curr_line_ptr; 
   int argc;
   char **argv;
   register char *p;
   register char *lp;
   register int count;
   register FILE *fp;
   time_t now;
   register int pri;           /* syslog priority */



   /*  there is no need to log the date and time 
       twice if using syslog */
   now=time ( (time_t) 0 );
   sprintf( locallogline, "%19.19s %12.12s sudo; ", ctime (&now), sudo_host );

   /* we need a pointer to the end of logline, and locallogline */
   p = logline + strlen(logline);  
   lp= locallogline +strlen(locallogline);

   getwd(cwd);                     /* so we know where we are... */

   switch ( code ) {

   case ALL_SYSTEMS_GO:
      sprintf ( p, "; Valid Access ; %s ; %s ; %s ; ", 
               sudo_user, entity_permitted, cwd );
      sprintf (lp, " Valid Access ; %s ; %s ; %s ; ", 
               sudo_user, entity_permitted, cwd );
#ifdef SYSLOG
      pri=Syslog_priority_OK;
#endif
      break;
      
   case VALIDATE_NO_USER:
   case LISTPRIVS_NO_USER:
      sprintf ( p, "; user NOT in sudoers ; %s ; ; %s ; ", 
               sudo_user, cwd );
      sprintf (lp, "user NOT in sudoers ; %s ; ; %s ; ", 
               sudo_user, cwd );
#ifdef SYSLOG
      pri=Syslog_priority_NO;
#endif
      break;
      
   case VALIDATE_NOT_OK: 
    case VALIDATE_NO_PERMISSION:
        sprintf ( p, "; Command NOT Allowed ; %s ; %s ; %s ; ", 
                    sudo_user, entity_depermitted, cwd );
        sprintf (lp, "Command NOT Allowed ; %s ; %s ; %s ; ", 
                    sudo_user, entity_depermitted, cwd );
#ifdef SYSLOG
        pri=Syslog_priority_NO;
#endif
	break;
	
    case VALIDATE_ERROR:
   case LISTPRIVS_ERROR:
	sprintf ( p, "; error in %s ; %s ; ; %s ; ", SUDOERS, sudo_user, cwd );
	sprintf (lp, "error in %s ; %s ; ; %s ; ", SUDOERS, sudo_user, cwd );
#ifdef SYSLOG
        pri=Syslog_priority_NO;
#endif
	break;

   case LISTPRIVS_PTS_ERROR:
      sprintf( p, "; Error in contacting PTS server ; %s ; %s ; %s ; ",
              sudo_user, entity_permitted, cwd );
      sprintf(lp, "Error in contacting PTS server ; %s ; %s ; %s ; ",
              sudo_user, entity_permitted, cwd );
#ifdef SYSLOG
      pri=Syslog_priority_NO;
#endif
      break;

   case GLOBAL_NO_PW_ENT:
        sprintf ( p, "; There is no /etc/passwd entry for uid ; %d ; ; %s ; ",
                  sudo_uid, cwd );
        sprintf (lp, "There is no /etc/passwd entry for uid ; %d ; ; %s ; ",
                 sudo_uid, cwd );
#ifdef SYSLOG
        pri=Syslog_priority_NO;
#endif
        break;

   case PASSWORD_NOT_CORRECT:
        sprintf ( p, "; %d incorrect passwords ; %s ; %s ; %s ; ", 
            TRIES_FOR_PASSWORD, sudo_user, entity_permitted, cwd );
        sprintf (lp, "%d incorrect passwords ; %s ; %s ; %s ; ", 
            TRIES_FOR_PASSWORD, sudo_user, entity_permitted, cwd );
#ifdef SYSLOG
        pri=Syslog_priority_NO;
#endif
	break;

   case GLOBAL_NO_HOSTNAME:
	strcat ( p, "; This machine does not have a hostname ; ; ; ; "  );
	strcat (lp, "This machine does not have a hostname ; ; ; ; " );
#ifdef SYSLOG
        pri=Syslog_priority_NO;
#endif
	break;

   case NO_SUDOERS_FILE:
      switch ( errno ) {
      case ENOENT:
	 sprintf ( p, "; There is no %s file ; %s ; ; %s ; ", 
                  SUDOERS, sudo_user, cwd );
	 sprintf (lp, "There is no %s file ; %s ; ; %s ; ", 
                  SUDOERS, sudo_user, cwd );
	 break;
      case EACCES:
	 sprintf ( p, "; %s needs to run setuid root ; %s ; ; %s ; ", 
                  Argv[0], sudo_user, cwd );
	 sprintf (lp, "%s needs to run setuid root ; %s ; ; %s ; ",
                  Argv[0], sudo_user, cwd );
	 break;
      default:
	 sprintf ( p, "; There is a problem opening %s ; %s ; ; %s ; ", 
                  SUDOERS, sudo_user, cwd );
	 sprintf (lp, "There is a problem opening %s ; %s ; ; %s ; ",
                  SUDOERS, sudo_user, cwd );
	 break;
      }
#ifdef SYSLOG
      pri=Syslog_priority_NO;
#endif
      break;

   case GLOBAL_HOST_UNREGISTERED:
      sprintf ( p, "; gethostbyname() cannot find host %s ; %s ; ; %s ; ", 
               sudo_host, sudo_user, cwd );
      sprintf (lp, "gethostbyname() cannot find host %s ; %s ; ; %s ; ", 
               sudo_host, sudo_user, cwd );
#ifdef SYSLOG
      pri=Syslog_priority_NO;
#endif
      break;

   default:
      sprintf( p + strlen(p) + 1 , "; found a wierd error ; %s ; %s ; %s ; ", 
              sudo_user, entity_permitted, cwd );
      sprintf(lp + strlen(lp)+ 1 , "found a wierd error ; %s ; %s ; %s ; ",
              sudo_user, entity_permitted, cwd );
#ifdef SYSLOG
      pri=Syslog_priority_NO;
#endif
      break;

   }
  

   /* if this error is from load_globals() don't put  argv in the message */
   if ( ! ( code & GLOBAL_PROBLEM ) ) {

      int local_count, local_argc; 
      char **local_argv;

      strcat ( logline, sudo_cmnd );  /* stuff the command into the logline */
      strcat ( logline, " ");
      strcat ( locallogline, sudo_cmnd );
      strcat ( locallogline, " " );

      argc = Argc-2;
      argv = Argv; argv++;
      p = logline + strlen(logline);
      lp= locallogline +strlen(locallogline);
      count =  (int)(logline + MAXLOGLEN - p);
 
      local_count = (int) (locallogline + MAXLOGLEN -lp );
      local_argc = argc; local_argv = argv; 

      /* now stuff as much of the rest of the line as will fit */
      while ( count > 0 && argc-- ) {
	 strncpy ( p, *++argv, count );
	 strcat ( p, " ");
	 p += 1 + (count < strlen(*argv) ? count : strlen(*argv) );
      strcat ( lp, " ..." );   /*  add an elipsis to the end */
	 count =  (int)(logline + MAXLOGLEN - p);
      }
      if ( count <= 0 )            /*  if the line is too long, */
	 strcat ( p, " ..." );   /*  add an elipsis to the end */
      strcat( p, "\n" );

      while ( local_count > 0 && local_argc-- ) {
	 strncpy ( lp, *++local_argv, local_count );
	 strcat ( lp, " ");
	 lp += 1 + (local_count < strlen(*local_argv) ? 
		    local_count : strlen(*local_argv) );
	 local_count =  (int)(locallogline + MAXLOGLEN - lp);
      }
      if ( local_count <= 0 )            /*  if the line is too long, */
          strcat ( lp, " ..." );   /*  add an elipsis to the end */
      strcat( lp, "\n" );
   }

   if ( appropriate(code) )
      send_mail();

#ifdef SYSLOG

   openlog ( Syslog_ident, Syslog_options, Syslog_facility );
   syslog( pri, logline ); 
   closelog();

#endif

   if( code == ALL_SYSTEMS_GO ) 
      logfile_name = LOGFILE;
   else
      logfile_name = FAILURE_LOGFILE;

   if ( (fp = fopen ( logfile_name, "a" )) == NULL )  {
      sprintf ( locallogline, "Can\'t open log file: %s", logfile_name );
      send_mail();
   }
   else {
      fprintf ( fp, "%s\n", locallogline );
      (void) fclose (fp);
   }
   
}




/**********************************************************************
 *
 *  send_mail()
 *
 *  This function attempts to mail to ALERTMAIL about the sudo error
 *
 */

char *exec_argv[]= { "sendmail" ,
			"-t" ,
			ALERTMAIL ,
			(char *) NULL };


static void send_mail()
{
   char *mailer=MAILER;
   char *subject=MAILSUBJECT;
   int fd[2];
   char buf[MAXLOGLEN+1024];

   if ( (mailer = find_path ( mailer )) == NULL ) {
      fprintf (stderr, "%s not found\n", mailer );
      exit (1);
   }

   signal ( SIGCHLD, reapchild );

   if ( fork () ) return;

   /* we don't want any security problems ... */
   setuid ( sudo_uid );

   signal ( SIGHUP, SIG_IGN );
   signal ( SIGINT, SIG_IGN );
   signal ( SIGQUIT, SIG_IGN );

   if ( pipe(fd) ) {
      perror( "send_mail: pipe" );
      exit ( 1 );
   }

   (void) dup2 ( fd[0], 0 );
   (void) dup2 ( fd[1], 1 );
   (void) close (fd[0]);
   (void) close (fd[1]);

   if ( ! fork () ) {


      /*      child                                parent     */

      (void) close(1);
      execv ( mailer, exec_argv );
      /* this should not happen */
      perror( "execv");
      exit (1);

   }

   else {

                                 (void) close(0);

                                 /* feed the data to sendmail */
                                 sprintf (buf, "To: %s\nSubject: %s\n\n%s\n\n",
					  ALERTMAIL, subject, locallogline );
                                 write ( 1, buf, strlen(buf));
                                 close ( 1 );

                                 exit ( 0 );
   }

}





/****************************************************************
 *
 *  reapchild()
 *
 *  This function gets rid fo all the ugly zombies
 */

static void reapchild ()
{
   (void) wait ( NULL );
}





/**********************************************************************
 *
 *  inform_user ()
 *
 *  This function lets the user know what is happening 
 *  when an error occurs
 */

void inform_user( code )
int code;
{

   switch ( code ) {
   case VALIDATE_NO_USER:
      fprintf( stderr,
      "%s is not in the sudoers file.  This incident will be reported.\n\n",
	      sudo_user );
      break;

   case LISTPRIVS_NO_USER: 
      fprintf( stderr, 
	      "%s has no sudo priveleges on this machine.\n\n", sudo_user );
      break;
      
   case VALIDATE_NOT_OK: 
   case VALIDATE_NO_PERMISSION:
      fprintf( stderr, 
	      "Sorry, user %s is not allowed to execute %s\n\n",
	      sudo_user, sudo_cmnd );
      break;
      
   case VALIDATE_ERROR:
   case LISTPRIVS_ERROR:
      fprintf( stderr, 
	      "Sorry, there is a fatal error in the sudoers file.\n\n" );
      break;
      
   case LISTPRIVS_PTS_ERROR:
      fprintf( stderr, 
	      "Sorry, Error in communicating with pts server.\n\n");
      break;
      
   case GLOBAL_NO_PW_ENT:
      fprintf ( stderr, 
	       "Intruder Alert!  You don\'t exist in the passwd file\n\n");
      break;
      
   case GLOBAL_NO_HOSTNAME:
      fprintf ( stderr, 
	       "This machine does not have a hostname\n\n" );
      break;
      
   case GLOBAL_HOST_UNREGISTERED:
      fprintf ( stderr, 
	       "This machine is not available via gethostbyname()\n\n");
      break;
      
   case PASSWORD_NOT_CORRECT:
      fprintf ( stderr, "Password not entered correctly after %d tries\n\n", 
	       TRIES_FOR_PASSWORD );
      break;
      
   default:
      fprintf ( stderr, 
	       "Something wierd happened.\n\n" );
      break;
      
   }
   
}





/****************************************************************
 *
 *  appropriate()
 *
 *  This function determines whether to send mail or not...
 */

static int appropriate( code )
int code;
{

   switch ( code ) {

      /*  these will NOT send mail  */
      
   case VALIDATE_OK: 
   case PASSWORD_NOT_CORRECT:
      return (0);
      break;

   case VALIDATE_NO_USER:
#ifdef SEND_MAIL_WHEN_NO_USER
      return (1);
#else
      return (0);
#endif
      break;

   case LISTPRIVS_NO_USER:
      return(0);
      break;

   case VALIDATE_NOT_OK:
   case VALIDATE_NO_PERMISSION:
#ifdef SEND_MAIL_WHEN_NOT_OK
      return (1);
#else
      return (0);
#endif
      break;

      /*  these WILL send mail  */

   case VALIDATE_ERROR: 
   case LISTPRIVS_ERROR:
   case NO_SUDOERS_FILE:
   case LISTPRIVS_PTS_ERROR:
   default:
      return (1);
      break;
      
   }
}





