#!/bin/sh
# the next line restarts using wish \
exec wish "$0" "$@"
#############################################
#            Tcl/Tk interface for Monte
#                 Yiping Zhan, 2002
# Modification history:
#	Sept 2002 - GSR added deuterium chemical shift flag
#       April 2003 - KH and GSR added new buttons for output display
#
# Parameter Array Description x()
#	x(0)    root filename
#       x(1)    subdirectory path
#	x(58)	flag for 2H chemical shift match
############################################
#
# default values for browser and postscript viewer
#   First pair are typical for IRIX/SGI
#   Second pair are typical for LINUX
set browser netscape
set psviewer xpsview
#set browser mozilla
#set psviewer kghostview
#
#
# Nothing below here should be changed
#
# default values for parameters
set default_values {rho sample/ 1 1 0 0 0 0 1 5.5 \
			0.0 0.0 0.0 0.0 0.0 \
			3.5 3.2 3.0 3.5 4.0 \
			0 0 0 0 0 0 0 0 0 \
			0 0 0 0 \
			0.10 0.05 0.10 0.10 0.10 0.10 0.10 0.10 \
			0 \
			sample/sample.out \
			0.30 0.30 0.30 0.30 0.30 \
			1 1 1 \
			1 \
			1 1 \
			60 1.0 1 \
			1 \
                        0}

# label names for parameters, some are not used in this program
set parnames {filename pathname pdb ss 3dhnoe 3dnnoe 4dnoe csl nsl cutoff \
		  "N off" "CO off" "CA off" "CB off" "CG off" \
		  N CO CA CB CG \
		  CO CA HA CB HB CG NNn NNc HH \
		  RT(i) RT(i-1) nsl csl \
		  N H CO CA HA CB HB CG \
		  matchold \
		  filenameo \
		  N H CO CA CB \
		  CO CA CB \
		  loglevel \
		  html ps \
		  psresiduewidth psscalingfactor psdisplaynoes \
		  numofcycles}

# simulated annealing parameter initial values
set sap_values {200 10.0 150.0 10000 4 1 0 0 0 0 0 0 \
		    150 10.0 100.0 10000 5 2 0 0 0 0 0 0 \
		    120 10.0 50.0 10000 10 3 0 5 5 0 5 0 \
		    50 10.0 20.0 20000 15 4 1 20 20 0 20 0 \
		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 \
 		    100 10.0 10.0 10000 4 1 0 0 0 0 0 0 }

# names for simulated annealing parameters
set sap_parnames {tstart tstep tfin swap_lim gamma swap_sz noe_scale CO CA HA CB HB}

# number of rows and columns for the chemical shift match tables
set nrowo 8
set ncolo 20
set nrow 20
set ncol 20

# rows for chemical shift match tables
set tablerows {N H CO CO CA CA HA1 HA2 HA1 HA2 CB CB HB1 \
		   HB2 HB1 HB2 CG CG Nn Nc}
set tablerowso {N H CO CO CA CA CB CB}

# maximum number of steps permitted for simulated annealing
set maxsaprownum 19

# pid list is initially empty
set pids {}


# set default values for all the parameters
proc set_defaults {} {
    global default_values
    global parnames
    global sap_values
    global saprownum
    global maxsaprownum
    global x
    global sap
    global cs
    global cso
    global nrowo
    global ncolo
    global nrow
    global ncol
    global fixed
    set saprownum 4
    for {set i 0} {$i<59} {incr i} {set x($i) [lindex $default_values $i]}
    for {set i 0} {$i<$maxsaprownum*12} {incr i} {set sap($i) [lindex $sap_values $i]}
    for {set i 0} {$i<$nrow*$ncol} {incr i} {set cs($i) 0}
    for {set i 0} {$i<$nrowo*$ncolo} {incr i} {set cso($i) 0}
    set cs(0) 1
    set cs(21) 1
    set cso(0) 1
    set cso(21) 1
    set fixed 1
}

# helper procedures for readparfile
# set values of arrays x, cs, cso and sap
proc setx {a b line} {
    global x
    for {set i $a} {$i<=$b} {incr i} {
	set x($i) [lindex $line [expr $i-$a]]
    }
}
proc setcs {a b line} {
    global cs
    for {set i $a} {$i<=$b} {incr i} {
	set cs($i) [lindex $line [expr $i-$a+1]]
    }
}
proc setcso {a b line} {
    global cso
    for {set i $a} {$i<=$b} {incr i} {
	set cso($i) [lindex $line [expr $i-$a+1]]
    }
}
proc setsap {a b line} {
    global sap
    for {set i $a} {$i<=$b} {incr i} {
	set sap($i) [lindex $line [expr $i-$a+1]]
    }
}

# read a parameter file
proc readparfile parinput {
    global nrow
    global ncol
    global nrowo
    global ncolo
    global maxsaprownum
    global saprownum
    global x
    global cs
    global cso
    set fileok 1
    if {$parinput == ""} return
    set input [open $parinput r]
    gets $input line
    gets $input line
    catch {set x(0) [lindex $line 0]}	;#root filename
    gets $input line
    gets $input line	
    catch {set x(1) [lindex $line 0]}	;#Subdirectory path:
    gets $input line
    gets $input line
    catch {set x(51) [lindex $line 0]}	;#Log level
    gets $input line
    gets $input line
    gets $input line
    catch {setx 2 8 $line}		;#Files available? ....
    gets $input line
    gets $input line
    catch {setx 52 53 $line}		;#HTML   PS
    gets $input line
    gets $input line
    catch {setx 54 56 $line}	    	;#residue_width   scale    NOE
    gets $input line
    gets $input line
    catch {set x(9) [lindex $line 0]}	;#Distance cut-off for NOESY (in Angstroms)
    gets $input line
    gets $input line
    gets $input line
    gets $input line			;#read in cs match table
    for {set i 0} {$i<$nrow} {incr i} {
	gets $input line
	catch {setcs [expr $i*$ncol] [expr ($i+1)*$ncol-1] $line}
    }
    gets $input line
    gets $input line
    gets $input line
    gets $input line
    catch {setx 10 14 $line}	;#Chemical shift offsets for Amino acid type determination
    gets $input line
    gets $input line
    gets $input line
    gets $input line
    catch {setx 15 19 $line}	;#residue type distribution width
    gets $input line
    gets $input line
    gets $input line
    catch {set x(58) [lindex $line 0]} ;#Deuterated Sample (0-no, 1-yes)
    gets $input line
    gets $input line
    gets $input line
    gets $input line
    catch {setx 20 28 $line}	;#Chemical shift matching - scaling factors
    gets $input line
    gets $input line
    catch {setx 29 32 $line}	;#rt_scale      rtmo_scale    nsl_scale  csl_scale
    gets $input line
    gets $input line
    gets $input line
    gets $input line
    catch {setx 33 40 $line}	;#Tolerances for Chemical Shift Matching...
    gets $input line
    gets $input line
    gets $input line
    catch {set x(41) [lindex $line 0]} ;#match old table
    gets $input line
    gets $input line
    gets $input line
    catch {set x(42) [lindex $line 0]} ;#old table filename
    gets $input line
    gets $input line
    gets $input line
    gets $input line			;#column def
    for {set i 0} {$i<$nrowo} {incr i} {
	gets $input line
	catch {setcso [expr $i*$ncolo] [expr ($i+1)*$ncolo-1] $line}
    }
    gets $input line
    gets $input line
    gets $input line
    gets $input line
    catch {setx 43 47 $line}	;#Matching old tolerances
    gets $input line
    gets $input line
    gets $input line
    gets $input line
    catch {setx 48 50 $line}	;#Scaling factors for old assignments
    gets $input line
    gets $input line
    gets $input line
    catch {set x(57) [lindex $line 0]} ;#numberofcycles
    gets $input line
    gets $input line
    gets $input line
    gets $input line		;#simulated annealing parameters
    set saprownum 0
    set tempvar notok
    for {set i 1} {$i<=$maxsaprownum} {incr i} {
	gets $input line
	catch {set tempvar [lindex $line 0]}
	if {$tempvar != $i && $tempvar != 0} {
	    set fileok 0
	    break
	} elseif {$tempvar == 0} break
	incr saprownum
	catch {setsap [expr ($i-1)*12] [expr ($i*12)-1] $line}
    }
    close $input
    for {set i 2} {$i<=8} {incr i} {
	if {$x($i) != 0 && $x($i) != 1} {
	    set fileok 0
	    break
	}
    }
    if {$x(51)!= 1 && $x(51)!= 2 && $x(51)!= 3} {set fileok 0}
    if {$x(52)!= 0 && $x(52) != 1} {set fileok 0}
    if {$x(53)!= 0 && $x(53) != 1} {set fileok 0}
    if {$x(56)!= 0 && $x(56) != 1} {set fileok 0}
    if {$x(41)!= 0 && $x(41) != 1 || $saprownum == 0} {set fileok 0}
    if {$x(4) != 0 && $x(5) == 1 || $x(4) == 1 && $x(6) == 1} {set fileok 0}
    for {set i 0} {$i<$nrow*$ncol} {incr i} {
	if {$cs($i) != 0 && $cs($i) != 1} {
	    set fileok 0
	    break
	}
    }
    for {set i 0} {$i<$nrowo*$ncolo} {incr i} {
	if {$cso($i) != 0 && $cso($i) != 1} {
	    set fileok 0
	    break
	}
    }
    if {$x(57)!= 1 && $x(57)!= 5 && $x(57)!= 10  && $x(57)!= 20 && $x(57)!= 30 \
	    && $x(57)!= 40 && $x(57)!= 50} {set fileok 0}
    if {!$fileok} {
	tk_dialog .dialog Error "$parinput is not of the expected parameter file format. \
        Parameters are set to default values." error 0 OK
	set_defaults
    }
    configold
    updatefilename
    config3dhnoe
    config4dnoe
    configps
    table_window .table 0
    packsapw
}

# draw label and entry rows in a text widget
proc drawle {w a b width} {
    global x
    global parnames
    $w tag configure center -justify center
    $w tag configure bar -font {-*-helvetica-bold-r-*-*-*-40-*-*-*-*-*-*}
    frame $w.fl$a
    frame $w.fe$a
    for {set i $a} {$i<=$b} {incr i} {
	label $w.fl$a.l$i -text [lindex $parnames $i] -bg gray -width $width
	entry $w.fe$a.e$i -textvariable x($i) -width $width
	grid $w.fl$a.l$i -row 0 -column [expr $i-$a] -padx 1
	grid $w.fe$a.e$i -row 1 -column [expr $i-$a]
    }
    $w insert end "\n" bar
    $w insert end " " center
    $w window create end -window $w.fl$a
    $w insert end "\n"
    $w insert end " " center
    $w window create end -window $w.fe$a
    $w insert end "\n"
}

# update all filenames according to the filename entry
proc updatefilename {} {
    global ttt
    global x
    $ttt.cs config -text "$x(0).cs"
    $ttt.seq config -text "$x(0).seq (sequence file)"
    $ttt.pdb config -text "$x(0).pdb (pdb file)"
    $ttt.ss config -text "$x(0).ss (secondary structure file)"
    $ttt.3dhnoe config -text "$x(0).3dh_noe (3D H-NH NOESY)"
    $ttt.3dnnoe config -text "$x(0).3dn_noe (3D N-NH NOESY)"
    $ttt.4dnoe config -text "$x(0).4dnoe (4D NH-NH NOESY)"
    $ttt.csl config -text "$x(0).csl"
    $ttt.nsl config -text "$x(0).nsl"  
}

# configure the states of other checkbutton(s) according to whether
# 3dhnoe file is selected
proc config3dhnoe {} {
    global x
    global ttt
    if {!$x(4)} {
	set x(5) 0
	$ttt.3dnnoe config -state disabled
	$ttt.4dnoe config -state normal
    } else {
	$ttt.3dnnoe config -state normal
	$ttt.4dnoe config -state disabled
    }
}

# configure the state of other checkbutton(s) according to whether
# 4dnoe file is selected
proc config4dnoe {} {
    global x
    global ttt
    if {$x(6)} {
	$ttt.3dhnoe config -state disabled
    } else {
	$ttt.3dhnoe config -state normal
    }
}

# configure the state of other checkbutton(s) and entries according to
# whether postscript output file is selected
proc configps {} {
    global x
    global ttt
    if {$x(53)} {
	$ttt.psresiduewidth config -fg black
	$ttt.psresiduewidthentry config -state normal -fg black
	$ttt.psscalingfactor config -fg black
	$ttt.psscalingfactorentry config -state normal -fg black
	$ttt.psdisplaynoes config -state normal
    } else {
	$ttt.psresiduewidth config -fg gray
	$ttt.psresiduewidthentry config -state disabled -fg gray
	$ttt.psscalingfactor config -fg gray
	$ttt.psscalingfactorentry config -state disabled -fg gray
	$ttt.psdisplaynoes config -state disabled
    }
}

# configure states of items according to whether match old table is selected
proc configold {} {
    global x
    global ttt
    if {$x(41)==1} {
	$ttt.matcholdb config -state normal
	$ttt.oldfilenamelabel config -fg black
	$ttt.oldfilenameentry config -state normal -fg black
	$ttt.oldfilebrowse config -state normal
	for {set i 43} {$i<=47} {incr i} {
	    $ttt.fl43.l$i config -fg black
	    $ttt.fe43.e$i config -state normal -fg black
	}
	for {set i 48} {$i<=50} {incr i} {
	    $ttt.fl48.l$i config -fg black
	    $ttt.fe48.e$i config -state normal -fg black
	}
	table_window .tableo 1
    } else {
	$ttt.matcholdb config -state disabled
	$ttt.oldfilenamelabel config -fg gray
	$ttt.oldfilenameentry config -state disabled -fg gray
	$ttt.oldfilebrowse config -state disabled
	for {set i 43} {$i<=47} {incr i} {
	    $ttt.fl43.l$i config -fg gray
	    $ttt.fe43.e$i config -state disabled -fg gray
	}
	for {set i 48} {$i<=50} {incr i} {
	    $ttt.fl48.l$i config -fg gray
	    $ttt.fe48.e$i config -state disabled -fg gray
	}
	catch {destroy .tableo}
    }
}

# helper procedures for writefile
# use regular expression matching to check whether an input is good
proc not_good_cutoff s {
    if {![regexp ^((\[1-9\])|(\[0-9\]\\.\[0-9\]*))$ $s]} {
	tk_dialog .dialog Error "$s is a bad cutoff value" error 0 OK
	return 1
    }
    return 0
}
proc not_integer s {
    if {[regexp ^((\[1-9\]\[0-9\]*)|0)$ $s]} {
	return 0
    } else {
	return 1
    }
}
proc not_some_number s {
    if {[regexp ^-?((\[1-9\]\[0-9\]*)|0|(\[0-9\]\\.\[0-9\]*)|((\[1-9\]\[0-9\]*)\\.\[0-9\]*))$ $s]} {
	return 0
    } else {
	return 1
    }
}
proc not_positive_number s {
    if {[regexp ^((\[1-9\]\[0-9\]*)|0|(\[0-9\]\\.\[0-9\]*)|((\[1-9\]\[0-9\]*)\\.\[0-9\]*))$ $s]} {
	return 0
    } else {
	return 1
    }
}

# helper procedure for writefile 
# check whether a file exists
proc file_not_exist s {
    if {![file exists $s]} {
	tk_dialog .dialog Error "File $s doesn't exist" error 0 OK
	return 1
    }
    return 0
}

# write a parameter file
proc writefile parfile {
    global x
    global sap
    global cs
    global cso
    global saprownum
    global parnames
    global nrowo
    global ncolo
    global nrow
    global ncol
    set err 0
    set war 0
    if {$parfile == ""} return
    if {[not_good_cutoff $x(9)]} {set err 1}
    if {![regexp ^.*/$ $x(1)]} {set x(1) "$x(1)/"}
    set fullname [file join $x(1) $x(0)]
    if {[file_not_exist $fullname.cs]} {set err 1}
    if {$x(2)} {
	if {[file_not_exist $fullname.pdb]} {set err 1}
    } else {
	if {[file_not_exist $fullname.seq]} {set err 1}
    }
    if {$x(3)} {
	if {[file_not_exist $fullname.ss]} {set err 1}
    }
    if {$x(4)} {
	if {[file_not_exist $fullname.3dh_noe]} {set err 1}
    }
    if {$x(5)} {
	if {[file_not_exist $fullname.3dn_noe]} {set err 1}
    }
    if {$x(6)} {
	if {[file_not_exist $fullname.4dnoe]} {set err 1}
    }
    if {$x(7)} {
	if {[file_not_exist $fullname.csl]} {set err 1}
    }
    if {$x(8)} {
	if {[file_not_exist $fullname.nsl]} {set err 1}
    }
    if {$x(41) && [file_not_exist $x(42)]} {set err 1}
    
    if {$x(53) && [not_integer $x(54)]} {
	tk_dialog .dialog Error "Bad value for residue width in output section" error 0 OK
	set err 1
    }
    if {$x(53) && [not_some_number $x(55)]} {
	tk_dialog .dialog Error "Bad value for scaling factor in output section" error 0 OK
	set err 1
    }
    if {$x(53) && ![not_some_number $x(55)]} {
	if {$x(55)<=0 || $x(55)>1} {
	    tk_dialog .dialog Error "Bad value for scaling factor in output section. \
            Some positive value not greater than 1 will be good." error 0 OK
	    set err 1
	}
    }
    for {set i 10} {$i<=14} {incr i} {
	if {[not_some_number $x($i)]} {
	    tk_dialog .dialog Error "Bad value for [lindex $parnames $i] \
            (Chemical shift offsets for amino acid type determination)" error 0 OK
	    set err 1
	}
    }
    for {set i 15} {$i<=19} {incr i} {
	if {[not_positive_number $x($i)]} {
	    tk_dialog .dialog Error "Bad value for [lindex $parnames $i] \
            (Residue type distribution width)" error 0 OK
	    set err 1
	}
    }
    for {set i 20} {$i<=32} {incr i} {
	if {[not_integer $x($i)]} {
	    tk_dialog .dialog Error "Bad value for [lindex $parnames $i] \
            (Chemical shift matching scaling factor)" error 0 OK
	    set err 1
	}
    }
    for {set i 33} {$i<=40} {incr i} {
	if {[not_positive_number $x($i)]} {
	    tk_dialog .dialog Error "Bad value for [lindex $parnames $i] \
	    (Tolerances for chemical shift matching)" error 0 OK
	    set err 1
	}
    }
    if {$x(41)} {
	for {set i 43} {$i<=47} {incr i} {
	    if {[not_positive_number $x($i)]} {
		tk_dialog .dialog Error "Bad value for [lindex $parnames $i] \
		(Matching old assignment Chemical shift tolerances)" error 0 OK
	    set err 1
	    }
	}
	for {set i 48} {$i<=50} {incr i} {
	    if {[not_integer $x($i)]} {
		tk_dialog .dialog Error "Bad value for [lindex $parnames $i] \
                (Scaling factor for old assignments)" error 0 OK
	    set err 1
	    }
	}
    }
    set flag1 1
    set flag2 1
    set flag3 1
    for {set i 0} {$i<12*$saprownum} {incr i} {
	if {$flag1 && ($i%12 == 1 || $i%12 == 2 || $i%12 == 6)} {
	    if {[not_positive_number $sap($i)]} {
		tk_dialog .dialog Error "Bad value for tstep, tfin or noe_scale \
                (Simulated annealing parameters)" error 0 OK
		set flag1 0
		set err 1
		continue
	    }
	} elseif {$flag2 && $i%12<7} {
	    if {[not_integer $sap($i)]} {
		tk_dialog .dialog Error "tstart, ncyclim, gamma and n_swap \
                should be non-negative integers" error 0 OK
		set flag2 0
		set err 1
		continue
	    }
	} elseif {$flag3 && [not_integer $sap($i)]} {
	    tk_dialog .dialog Error "Repulsive terms should be non-negative integers" \
		error 0 OK
	    set flag3 0
	    set err 1
	    continue
	}
    }
    if {($x(7) == 0) && ($x(32) != 0)} {
	tk_dialog .dialog Warning "Since a csl file is not selected, \
        the non-zero value for the csl scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(8) == 0) && ($x(31) != 0)} {
	tk_dialog .dialog Warning "Since an nsl file is not selected, \
        the non-zero value for the nsl scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    for {set i 0} {$i<$nrow} {incr i} {
	set r($i) 0
	for {set j [expr $i*$ncol]} {$j<[expr ($i+1)*$ncol]} {incr j} {
	    if {$cs($j)} {set r($i) 1; break}
	}
    }
    for {set i 0} {$i<$nrowo} {incr i} {
	set ro($i) 0
	for {set j [expr $i*$ncolo]} {$j<[expr ($i+1)*$ncolo]} {incr j} {
	    if {$cso($j)} {set ro($i) 1; break}
	}
    }
    if {!($r(0) && $r(1))} {
	tk_dialog .dialog Error "Both N and H have to be selected in the CS ID \
        table" error 0 OK
	set err 1
    }
    if {($x(20) != 0) && !($r(2) && $r(3))} {
	tk_dialog .dialog Warning "Since  CO and CO are not both selected in the CS ID \
        table, the non-zero value for the CO scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(21) != 0) && !($r(4) && $r(5))} {
	tk_dialog .dialog Warning "Since CA and CA are not both selected in the CS ID \
        table, the non-zero value for the CA scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(22) != 0) && !($r(6) && $r(8))} {
	tk_dialog .dialog Warning "Since HA1 and HA1 are not both selected in the CS ID \
        table, the non-zero value for the HA scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(23) != 0) && !($r(10) && $r(11))} {
	tk_dialog .dialog Warning "Since CB and CB are not both selected in the CS ID \
        table, the non-zero value for the CB scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(24) != 0) && !($r(12) && $r(13) && $r(14) && $r(15))} {
	tk_dialog .dialog Warning "Since HB1, HB2, HB1 and HB2 are not all selected in the CS ID \
        table, the non-zero value for the HB scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(25) != 0) && !($r(16) && $r(17))} {
	tk_dialog .dialog Warning "Since CG and CG are not both selected in the CS ID \
        table, the non-zero value for the CG scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(26) != 0) && !$r(18)} {
	tk_dialog .dialog Warning "Since Nn is not selected in the CS ID \
        table, the non-zero value for the NNn scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {($x(27) != 0) && !$r(19)} {
	tk_dialog .dialog Warning "Since Nc is not selected in the CS ID \
        table, the non-zero value for the NNc scaling factor is going to be ignored" warning 0 OK
	set war 1
    }
    if {$x(41) && !($ro(0) && $ro(1))} {
	tk_dialog .dialog Error "Both N and H have to be selected in the old CS ID \
        table" error 0 OK
	set err 1
    }
    if {$x(41) && ($x(48) != 0) && !($ro(2) && $r(2)) && !($ro(3) && $r(3))} {
	tk_dialog .dialog Warning "There is no pair of old and new CO or CO to match according to \
        the CS ID tables. The non-zero value for the CO scaling factor (for old assignments) \
        is going to be ignored" warning 0 OK
	set war 1
    }
    if {$x(41) && ($x(49) != 0) && !($ro(4) && $r(4)) && !($ro(5) && $r(5))} {
	tk_dialog .dialog Warning "There is no pair of old and new CA or CA to match according to \
        the CS ID tables. The non-zero value for the CA scaling factor (for old assignments) \
        is going to be ignored" warning 0 OK
	set war 1
    }
    if {$x(41) && ($x(50) != 0) && !($ro(6) && $r(10)) && !($ro(7) && $r(11))} {
	tk_dialog .dialog Warning "There is no pair of old and new CB or CB to match according to \
        the CS ID tables. The non-zero value for the CB scaling factor (for old assignments) \
        is going to be ignored" warning 0 OK
	set war 1
    }
    if {$err || $war} {
	tk_messageBox -type ok -message "Parameter file is saved in spite of \
        previous error or warning message(s)"
    }
    set cscol {"N     " "H     " "CO    " "CO-1  " "CA    " "CA-1  " "HA1   " \
		   "HA2   " "HA1-1 " "HA2-1 " "CB    " "CB-1  " "HB1   " "HB2   " \
		   "HB1-1 " "HB2-1 " "CG    " "CG-1  " "Nn    " "Nc    "}
    set csocol {"N     " "H     " "CO    " "CO-1  " "CA    " "CA-1  " "CB    " "CB-1  "}
    set output [open $parfile w]
    puts $output "Root Filename:"
    puts $output "$x(0)"
    puts $output "Subdirectory path:"
    puts $output "$x(1)"
    puts $output "Log level"
    puts $output "$x(51)"
    puts $output "Files available? (1=yes,0=no)"
    puts $output "(pdb)  (2'struct)  (3DhNOE) (3DnNOE) (4DNOE) (Cslab) (Nslab)"
    puts $output "$x(2)       $x(3)           $x(4)       $x(5)        $x(6)        $x(7)      $x(8)"
    puts $output "HTML   PS"
    puts $output "$x(52)      $x(53)"
    puts $output "residue_width   scale    NOE"
    puts $output "$x(54)             $x(55)        $x(56)"
    puts $output "Distance cut-off for NOESY (in Angstroms)"
    puts $output "$x(9)"
    puts $output ""
    puts $output "chemical shift table column definitions"
    puts $output "      1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20"
    puts $output "----------------------------------------------------------------"
    for {set i 0} {$i<$nrow*$ncol} {incr i} {
	if {$i%$ncol == 0} {
	    puts -nonewline $output "[lindex $cscol [expr $i/$ncol]]"
	    puts -nonewline $output "$cs($i)"
	} elseif {$i%$ncol == $ncol-1} {
	    puts -nonewline $output "  $cs($i)"
	    puts $output ""
	} else {
	    puts -nonewline $output "  $cs($i)"
	}
    }
    puts $output ""
    puts $output "Chemical shift offsets for Amino acid type determination"
    puts $output "N_off    CO_off   CA_off    CB_off    CG_off"
    puts $output "   $x(10)     $x(11)      $x(12)        $x(13)        $x(14)"
    puts $output ""
    puts $output "residue type distribution width"
    puts $output "N      CO     CA      CB      CG"
    puts $output "$x(15)    $x(16)    $x(17)     $x(18)     $x(19)"
    puts $output ""
    puts $output "Deuterated Sample (0-no, 1-yes)"
    puts $output "$x(58)"
    puts $output ""
    puts $output "Chemical shift matching - scaling factors (must be integers)"
    puts $output "CO   CA   HA   CB   HB   CG   NNn  NNc  HH"
    puts $output " $x(20)    $x(21)    $x(22)    $x(23)    $x(24)    $x(25)    $x(26)    $x(27)    $x(28)"
    puts $output "rt_scale      rtmo_scale    nsl_scale  csl_scale"
    puts $output "$x(29)             $x(30)             $x(31)          $x(32)"
    puts $output ""
    puts $output "Tolerances for Chemical Shift Matching \[sigma (ppm) for Gaussian\]"
    puts $output " HN    H      CO     CA     HA     CB     HB     CG"
    puts $output "$x(33)   $x(34)   $x(35)   $x(36)   $x(37)   $x(38)   $x(39)   $x(40)"
    puts $output ""
    puts $output "match old table(1=yes,0=no)"
    puts $output "$x(41)"
    puts $output ""
    puts $output "old table filename (include subdirectory path i.e. directory/file)"
    puts $output "$x(42)"
    puts $output ""
    puts $output "Chemical shift table column definitions for old solution"
    puts $output "      1  2  3  4  5  6  7  8  9 10 11 12 13 14 15 16 17 18 19 20"
    puts $output "----------------------------------------------------------------"
    for {set i 0} {$i<$nrowo*$ncolo} {incr i} {
	if {$i%$ncolo == 0} {
	    puts -nonewline $output "[lindex $csocol [expr $i/$ncolo]]"
	    puts -nonewline $output "$cso($i)"
	} elseif {$i%$ncolo == $ncolo-1} {
	    puts -nonewline $output "  $cso($i)"
	    puts $output ""
	} else {
	    puts -nonewline $output "  $cso($i)"
	}
    }
    puts $output ""
    puts $output "Matching old assignment chemical shift tolerances (sigma in ppm)"
    puts $output "notol  hnotol  cootol  caotol  cbotol"
    puts $output "$x(43)    $x(44)   $x(45)    $x(46)    $x(47)"
    puts $output ""
    puts $output "Scaling factors (must be integers) for old assignments"
    puts $output "COold  CAold  CBold"
    puts $output "  $x(48)     $x(49)      $x(50)"
    puts $output ""
    puts $output numberofcycles
    puts $output $x(57)
    puts $output ""
    puts $output "		simulated annealing parameters"
    puts $output "# tstart tstep   tfin nswaplim  gamma n_swap noe_scale   Repulsive Terms "
    puts $output "  |       |      |     |          |     |     |        CO  CA  HA  CB  HB"
    for {set i 1} {$i<=$saprownum} {incr i} {
	puts -nonewline $output "$i"
	for {set j 0} {$j<12} {incr j} {
	    set sub [expr ($i-1)*12+$j]
	    puts -nonewline $output "    $sap($sub)"
	}
	puts $output ""
    }
    puts $output "0    0     0.0    0.0  0         0      0     0         0   0   0   0   0"
    close $output
}

# configure the states of all the checkbuttons in a chemical shift match table
proc configcks {w o} {
    global nrow
    global ncol
    global cs
    global nrowo
    global ncolo
    global cso
    global tablerows
    global tablerowso
    if {$o} {
	for {set i 0} {$i<$nrowo} {incr i} {
	    set r($i) 0
	    for {set j [expr $i*$ncolo]} {$j<[expr ($i+1)*$ncolo]} {incr j} {
		if {$cso($j)} {set r($i) 1; break}
	    }
	}
	for {set i 0} {$i<$ncolo} {incr i} {
	    set c($i) 0
	    for {set j $i} {$j<=$i+($nrowo-1)*$ncolo} {set j [expr $j+$ncolo]} {
		if {$cso($j)} {
		    set c($i) 1
		    $w.is[expr $i+1] config -text [lindex $tablerowso [expr $j/$ncolo]]	    
		    break
		}
	    }
	    if {!$c($i)} {$w.is[expr $i+1] config -text ""}
	}
	for {set i 0} {$i<$nrowo*$ncolo} {incr i} {
	    set rnum [expr $i/$ncolo]
	    set cnum [expr $i%$ncolo]
	    if {($rnum<2 && $cnum<2) || ($rnum>1 && $cnum>1)} {
		if {($r($rnum) || $c($cnum)) && !$cso($i)} {
		    $w.cs$i configure -state disabled
		} else {
		    $w.cs$i configure -state normal
		}
	    }
	}
    } else {
	for {set i 0} {$i<$nrow} {incr i} {
	    set r($i) 0
	    for {set j [expr $i*$ncol]} {$j<[expr ($i+1)*$ncol]} {incr j} {
		if {$cs($j)} {set r($i) 1; break}
	    }
	}
	for {set i 0} {$i<$ncol} {incr i} {
	    set c($i) 0
	    for {set j $i} {$j<=$i+($nrow-1)*$ncol} {set j [expr $j+$ncol]} {
		if {$cs($j)} {
		    set c($i) 1
		    $w.is[expr $i+1] config -text [lindex $tablerows [expr $j/$ncol]]
		    break
		}
	    }
	    if {!$c($i)} {$w.is[expr $i+1] config -text ""}
	}
	for {set i 0} {$i<$nrow*$ncol} {incr i} {
	    set rnum [expr $i/$ncol]
	    set cnum [expr $i%$ncol]
	    if {($rnum<2 && $cnum<2) || ($rnum>1 && $cnum>1)} {
		if {($r($rnum) || $c($cnum)) && !$cs($i)} {
		    $w.cs$i configure -state disabled
		} else {
		    $w.cs$i configure -state normal
		}
	    }
	}
    }
}

# clear a chemical shift identification table
proc clear_cs_table {w o} {
    global nrow
    global ncol
    global cs
    global nrowo
    global ncolo
    global cso
    if {$o} {
	for {set i 0} {$i<$ncolo*$nrowo} {incr i} {set cso($i) 0}
    } else {
	for {set i 0} {$i<$ncol*$nrow} {incr i} {set cs($i) 0}
    }
    configcks $w $o
}

# open a chemical shift identification table
proc table_window {w o} {
    catch {destroy $w}
    toplevel $w
    frame $w.buttons
    pack $w.buttons -side bottom -fill x -pady 2m
    button $w.buttons.clear -text "clear all" -command "clear_cs_table $w.f $o"
    button $w.buttons.close -text close -command "destroy $w"
    pack $w.buttons.clear $w.buttons.close -side left -expand 1
    if {$o} {
	wm title $w "Chemical shift identification table -- old output"
    } else {
	wm title $w "Chemical shift identification table"
    }
    pack [frame $w.f]
    table_contents $w.f $o
    configcks $w.f $o
}

# helper procedure for table_window
# draw the table contents
proc table_contents {w o} {
    global nrow
    global ncol
    global cs
    global tablerows
    global nrowo
    global ncolo
    global cso
    global tablerowso
    if {$o} {
	label $w.colnum -text "Column \#"
	grid $w.colnum -row 0 -column 0
	for {set i 1} {$i<=$ncolo} {incr i} {
	    label $w.tablecol$i -text "$i"
	    label $w.is$i -text ""
	    grid $w.tablecol$i -row 0 -column $i
	    grid $w.is$i -row 1 -column $i
	}
	for {set i 1} {$i<=$nrowo} {incr i} {
	    label $w.tablerow$i -text "[lindex $tablerowso [expr $i-1]]"
	    grid $w.tablerow$i -row [expr $i+1] -column 0
	}
	for {set i 0} {$i<$ncolo*$nrowo} {incr i} {
	    set rnum [expr $i/$ncolo]
	    set cnum [expr $i%$ncolo]
	    if {($rnum<2 && $cnum<2) || ($rnum>1 && $cnum>1)} {
		checkbutton $w.cs$i -variable cso($i) -command "configcks $w 1" -width 2
		grid $w.cs$i -row [expr $i/$ncolo+2] -column [expr $i%$ncolo+1]
	    }
	}
    } else {
	label $w.colnum -text "Column \#"
	grid $w.colnum -row 0 -column 0
	for {set i 1} {$i<=$ncol} {incr i} {
	    label $w.tablecol$i -text "$i"
	    label $w.is$i -text ""
	    grid $w.tablecol$i -row 0 -column $i
	    grid $w.is$i -row 1 -column $i
	}
	for {set i 1} {$i<=$nrow} {incr i} {
	    label $w.tablerow$i -text "[lindex $tablerows [expr $i-1]]"
	    grid $w.tablerow$i -row [expr $i+1] -column 0
	}
	for {set i 0} {$i<$ncol*$nrow} {incr i} {
	    set rnum [expr $i/$ncolo]
	    set cnum [expr $i%$ncolo]
	    if {($rnum<2 && $cnum<2) || ($rnum>1 && $cnum>1)} {
		checkbutton $w.cs$i -variable cs($i) -command "configcks $w 0" -width 2
		grid $w.cs$i -row [expr $i/$ncol+2] -column [expr $i%$ncol+1]
	    }
	}
    }
}

# add a row in the simulated annealing parameter window
proc addrow {} {
    global saprownum
    global maxsaprownum
    if {$saprownum == $maxsaprownum} {
	tk_dialog .dialog Error "Maximum number of rows is $maxsaprownum!" error 0 OK
	return
    }
    incr saprownum
    packsapw
}

# delete a row in the simulated annealing parameter window
proc deleterow {} {
    global saprownum
    if {$saprownum == 1} {
	tk_dialog .dialog Error "Minimum number of rows is 1!" error 0 OK
	return
    }
    set saprownum [expr $saprownum-1]
    packsapw
}

# pack the simulated annealing paramter window
proc packsapw {} {
    global saprownum
    global sap_parnames
    global sap_values
    global ts
    set sapwidth {5 5 6 7 6 7 8 3 3 3 3 3}
    catch {destroy $ts}
    text $ts -xscrollcommand ".sapw.scrollx set" -yscrollcommand ".sapw.scrolly set" -height 12 \
	-width 56 -highlightthickness 0 -borderwidth 0 -wrap none -font {-*-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*}
    pack $ts -expand yes -fill both
    $ts tag configure bar -font {-*-helvetica-bold-r-*-*-*-40-*-*-*-*-*-*}
    $ts tag configure center -justify center
    $ts insert end "\n"
    $ts insert end "Simulated annealing parameters\n" center
    $ts insert end "\n" bar
    label $ts.numofcycleslable -text "Number of cycles" -width 25 -bg gray
    tk_optionMenu $ts.numofcycles x(57) 1 5 10 20 30 40 50
    $ts insert end "\n"
    $ts insert end " " center
    $ts window create end -window $ts.numofcycleslable -padx 3
    $ts window create end -window $ts.numofcycles -padx 3
    $ts insert end "\n"
    label $ts.blank -text "" -width 56
    label $ts.repulsive -text "Repulsive terms" -bg gray -width 20
    $ts insert end " " center
    $ts window create end -window $ts.blank
    $ts window create end -window $ts.repulsive
    $ts insert end "\n"
    $ts insert end " " center
    label $ts.row0 -text Step -width 4
    $ts window create end -window $ts.row0
    for {set i 0} {$i<12} {incr i} {
	label $ts.l$i -text [lindex $sap_parnames $i] -bg gray -width [lindex $sapwidth $i]
	$ts window create end -window $ts.l$i -padx 1
    }
    for {set i 1} {$i<=$saprownum} {incr i} {
	$ts insert end "\n"
	label $ts.row$i -text $i -width 4
	$ts insert end " " center
	$ts window create end -window $ts.row$i
	for {set j 0} {$j<12} {incr j} {
	    set a [expr ($i-1)*12+$j]
	    entry $ts.e$a -textvariable sap($a) -width [lindex $sapwidth $j]
	    $ts window create end -window $ts.e$a
	}
    }
    bind $ts <Motion> {break}
    $ts configure -state disabled
}

proc killall {} {
    global pids
    foreach i $pids {
 	exec kill -9 $i > .temp 2> .temp &
    }
    set pids {}
}

set_defaults
wm title . "Monte v2.02"

#DEFINE MAINFRAME
#Begin with text box (.mainframe.text)

frame .mainframe -highlightthickness 2 -borderwidth 2 -relief sunken
set ttt .mainframe.text
text $ttt -yscrollcommand ".scroll set" -height 25 -width 56 \
    -highlightthickness 0 -borderwidth 0 -font {-*-helvetica-bold-r-*-*-*-140-*-*-*-*-*-*}
scrollbar .scroll -troughcolor yellow -highlightbackground SteelBlue -highlightthickness 2 -command "$ttt yview"

#Define Formating Macros For Text

$ttt tag configure title -font {-*-helvetica-bold-r-*-*-*-120-*-*-*-*-*-*} -justify center
$ttt tag configure center -justify center
$ttt tag configure bar -font {-*-helvetica-bold-r-*-*-*-40-*-*-*-*-*-*}

#Define widgets for filename, pathname, log level
 
label $ttt.filenamelabel -text "Root Filename" -width 25 -bg gray
entry $ttt.filenameentry -textvariable x(0) -width 25
bind $ttt.filenameentry <KeyRelease> updatefilename

label $ttt.pathnamelabel -text Pathname -width 25 -bg gray
entry $ttt.pathnameentry -textvariable x(1) -width 25

label $ttt.loglevellabel -text "Log level" -width 25 -bg gray
tk_optionMenu $ttt.loglevelmenu x(51) 1 2 3
label $ttt.loglevelblank -text "" -width 17

#Define widgets for files available

checkbutton $ttt.cs -text "filename.cs" -variable fixed \
    -state disabled -disabledforeground black -width 36 -anchor w  -cursor "left_ptr"
radiobutton $ttt.seq -text "filename.seq (sequence file)" -variable x(2) -value 0 \
    -width 36 -anchor w -cursor "left_ptr"
radiobutton $ttt.pdb -text "filename.pdb (pdb file)" -variable x(2) -value 1 \
    -width 36 -anchor w -cursor "left_ptr"
checkbutton $ttt.ss -text "filename.ss (secondary structure file)" -variable x(3) \
    -width 36 -anchor w -cursor "left_ptr"
checkbutton $ttt.3dhnoe -text "filename.3dh_noe (3D H-NH NOESY)" -variable x(4) \
    -command config3dhnoe -width 36 -anchor w -cursor "left_ptr"
checkbutton $ttt.3dnnoe -text "filename.3dn_noe (3D H-NH NOESY)" -variable x(5) \
    -width 36 -anchor w -cursor "left_ptr"
checkbutton $ttt.4dnoe -text "filename.4dnoe (4D NH-NH NOESY)" -variable x(6) \
    -width 36 -anchor w -cursor "left_ptr" -command config4dnoe
checkbutton $ttt.csl -text "filename.csl" -variable x(7) -width 36 -anchor w -cursor "left_ptr"
checkbutton $ttt.nsl -text "filename.nsl" -variable x(8) -width 36 -anchor w -cursor "left_ptr"

#Define widgets for output parameters

checkbutton $ttt.html -text HTML -variable x(52) -width 36 -anchor w -cursor "left_ptr"
checkbutton $ttt.ps -text Postscript -variable x(53) -width 36 \
    -command configps -anchor w -cursor "left_ptr"

label $ttt.psresiduewidth -text "Residue width" -width 15 -bg azure
entry $ttt.psresiduewidthentry -textvariable x(54) -width 15

label $ttt.psscalingfactor -text "Scaling factor" -width 15 -bg azure
entry $ttt.psscalingfactorentry -textvariable x(55) -width 15

checkbutton $ttt.psdisplaynoes -text "Display NOE's" -variable x(56) \
    -width 36 -anchor w -cursor "left_ptr"

label $ttt.cutoff -text "Distance cutoff ()" -width 25 -bg gray
entry $ttt.cutoffentry -textvariable x(9) -width 20

#Define widget for 1H/2H residue chemical shifts
radiobutton $ttt.deut_cd -text "CD shifts" -variable x(58) -value 1 \
    -width 15 -anchor w -cursor "left_ptr"
radiobutton $ttt.deut_ch -text "CH shifts" -variable x(58) -value 0 \
    -width 15 -anchor w -cursor "left_ptr"


#THIS CREATES THE UPPER PART OF THE WINDOW

$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.filenamelabel -padx 3
$ttt window create end -window $ttt.filenameentry -padx 3
$ttt insert end "\n"    
$ttt insert end " " center

$ttt window create end -window $ttt.pathnamelabel -padx 3
$ttt window create end -window $ttt.pathnameentry -padx 3
$ttt insert end "\n"
$ttt insert end " " center

$ttt window create end -window $ttt.loglevellabel -padx 3
$ttt window create end -window $ttt.loglevelmenu -padx 3
$ttt window create end -window $ttt.loglevelblank -padx 3
$ttt insert end "\n\n"

$ttt insert end "Files available" title
$ttt insert end "\n"
$ttt insert end "\n" bar
$ttt insert end " " center

$ttt window create end -window $ttt.cs
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.seq
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.pdb
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.ss
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.3dhnoe
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.3dnnoe
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.4dnoe
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.csl
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.nsl

$ttt insert end "\n\n"
$ttt insert end "Output" title
$ttt insert end "\n"
$ttt insert end "\n" bar
$ttt insert end " " center
$ttt window create end -window $ttt.html
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.ps
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.psresiduewidth -padx 3
$ttt window create end -window $ttt.psresiduewidthentry -padx 3
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.psscalingfactor -padx 3
$ttt window create end -window $ttt.psscalingfactorentry -padx 3
$ttt insert end "\n"
$ttt insert end " " center
$ttt window create end -window $ttt.psdisplaynoes

$ttt insert end "\n"
$ttt insert end "\n_________________________________________________________________" title
$ttt insert end "\n\n"
$ttt insert end " " center
$ttt window create end -window $ttt.cutoff -padx 3
$ttt window create end -window $ttt.cutoffentry -padx 3
$ttt insert end "\n\n"

$ttt insert end "\n\nResidue type distribution width (sigma in ppm)\n" title
drawle $ttt 15 19 10

$ttt insert end " " center
$ttt window create end -window $ttt.deut_cd -padx 3
$ttt window create end -window $ttt.deut_ch -padx 3

$ttt insert end "\n\nChemical shift matching - scaling factors (must be integers)\n" title
drawle $ttt 20 27 6
drawle $ttt 29 32 12
$ttt insert end "\n\nTolerances for chemical shift matching \[sigma (ppm) for Gaussian\]\n" title
drawle $ttt 33 40 6

$ttt insert end "\nChemical shift offsets for amino acid type determination\n" title
drawle $ttt 10 14 10


$ttt insert end "\n_________________________________________________________________" title

#DEFINE WIDGETS FOR OLD OUTPUT MATCHING

checkbutton $ttt.matchold -text "match old output" -variable x(41) \
    -command configold -cursor "left_ptr"
button $ttt.matcholdb -text "Open old CS ID table" -command {table_window .tableo 1}
label $ttt.oldfilenamelabel -text "Old output filename (include path)" -width 28 -bg gray
entry $ttt.oldfilenameentry -textvariable x(42) -width 20
button $ttt.oldfilebrowse -text "Browse ..." -command {set x(42) \
     [tk_getOpenFile -filetypes {{{Old output file} {.out}} {{All Files} *}}]}

#Insert them into the text box

$ttt insert end "\n\n"
$ttt insert end " " center

$ttt window create end -window $ttt.matchold -padx 3
$ttt window create end -window $ttt.matcholdb -padx 3
$ttt insert end "\n\n"
$ttt insert end " " center
$ttt window create end -window $ttt.oldfilenamelabel -padx 3
$ttt window create end -window $ttt.oldfilenameentry
$ttt window create end -window $ttt.oldfilebrowse

$ttt insert end "\n\n\nMatching old assignment Chemical shift tolerances (sigma in ppm)\n" title
drawle $ttt 43 47 10
$ttt insert end "\n\nScaling factors (must be integers) for old assignments\n" title
drawle $ttt 48 50 17

bind $ttt <Motion> {break}
$ttt configure -state disabled

#DEFINE LOWER BUTTONS IN MAIN FRAME

frame .fb

#Define button widgets

button .fb.b0 -text "Open CS ID table" -command {table_window .table 0}
button .fb.b1 -text "Open par file" -command \
    {readparfile [tk_getOpenFile -filetypes \
		      {{{Parameter Files} {.par}} {{All Files} *}}]}
button .fb.b2 -text "Save par file" -command \
    {writefile [tk_getSaveFile -defaultextension .par -initialfile Untitled \
		    -filetypes {{{Parameter Files} {.par}} {{All Files} *}}]}
button .fb.b3 -text "Start" -command \
    {set parfile [tk_getOpenFile -filetypes \
 		      {{{Parameter Files} {.par}} {{All Files} *}}];
 	if {$parfile != ""} {lappend pids [exec ./fm.exe $parfile &]}}
button .fb.b4 -text "Kill" -command killall
button .fb.b5 -text Quit -command {destroy .}
pack .fb.b0 .fb.b1 .fb.b2 .fb.b3 .fb.b4 .fb.b5 -side left -expand 1

frame .fc
set curdirect [pwd]
button .fc.b1 -text "HTML" -command {exec $browser $curdirect/$x(1)main.html &}
button .fc.b2 -text "Corr Plot" -command {exec $psviewer $curdirect/$x(1)solutions/corr.ps &}
button .fc.b3 -text "Solution Output" -command {exec $psviewer $curdirect/$x(1)$x(0)_out.ps &}
pack .fc.b1 .fc.b2 .fc.b3 -side left -expand 1

#DEFINE SIMULATED ANNEALING FRAMESET

frame .sapw -highlightthickness 2 -borderwidth 2 -relief sunken
frame .sapw.b

pack .sapw.b -side bottom -fill x -pady 1

button .sapw.b.add -text "Add a row" -command addrow -width 14
button .sapw.b.delete -text "Delete last row" -command deleterow -width 14
pack .sapw.b.add .sapw.b.delete -side left -expand 1;# -fill both

set ts .sapw.f.text

frame .sapw.f
scrollbar .sapw.scrollx -orient horizontal -command "$ts xview"
scrollbar .sapw.scrolly -orient vertical -command "$ts yview"
pack .sapw.scrollx -side bottom -fill x
pack .sapw.scrolly -side right -fill y
pack .sapw.f -expand yes -fill both


pack .fb -side bottom -fill x -pady 2m
pack .fc -side bottom -fill x -pady 2m

pack .sapw -side bottom -fill x
pack .scroll -side right -fill y
pack $ttt -expand yes -fill both
pack .mainframe -expand yes -fill both

configold
updatefilename
config3dhnoe
config4dnoe
configps
table_window .table 0
packsapw
