/*
** Copyright (C) 2000 Roman Danyliw <roman@danyliw.com>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* Snort rules file analyzer which with the help of nmap finds 
 *   all the rules which do NOT apply to the local network
 *
 * Author: Roman Danyliw <roman@danyliw.com>
 *
 * Contributions: Jed Pickel <jed@pickel.net> for the idea
 *  
 * See http://www.andrew.cmu.edu/~rdanyliw/snort/ for the latest
 * version and documentation.
 *
 * file: snortrules.c 
 * version: 0.5
 * 
 * Purpose:
 *
 *    Many of the repositories for Snort rules may include those that
 *    do not apply to your network.  Therefore, this code was written
 *    to identify (and possibly discard) in a semi-automated fashion
 *    the subset of rules which actually apply to the local network.
 *    It essentially uses nmap (www.insecure.org) to initally map which 
 *    services are present.  Then with a list of open TCP and UDP ports, 
 *    snortrule determine which rules do not apply. (see the README for
 *    more details)
 *
 *    When a rule is determined to be non-applicable, two actions can 
 *    be taken.  A simple approach is to take the stance that the 
 *    potential exploit is a non-issue because it is targeted at a 
 *    non-existent service (i.e. you are not vulnerable).  Thus, 
 *    non-applicable rules can be discarded from the rules list.  However, 
 *    for the more paranoid, it is valuable to know _all_ the attacks 
 *    which are being attempted against your network.  Therefore, snortrules 
 *    can also merely change the text of the particular alert to indicate 
 *    that the rule was triggered but does not apply.
 *
 * Prerequisite:
 *
 *    - nmap 2.12 or greater
 *
 * Usage
 *    
 *    nmap [parameters] | 
 *      snortrules -a[x|f] -net[v] <variable> -in <filename> -out <filename> 
 *
 *    -a[x|f]            : action to take for non-applicable rule
 *                          -ax : discard the rule
 *                          -af : keep the rule, but flag the alert message
 *    -net[v] <variable> : name denoting the internal networki (e.g. any, 127.0.0.1/24)
 *                          -netv : name is a variable (e.g. INTERNAL, _not_ $INTERNAL)
 *    -in <filename>     : unparsed Snort rules file
 *    -out <filename>    : output file to be created with parsed rules
 *    -help              : displays this help and exits
 *
 * Example:
 *    nmap -sT -sU localhost | \
 *    snortrules -ax -net INTERNAL -in vision.conf -out myrules.conf
 *
 *    Run nmap on the localhost searching for open TCP and UDP ports (-sT -SU).
 *    Read rules from 'vision.conf' (-in vision.conf), using 'INTERNAL' as the
 *    variable representing the internal network (-net INTERNAL).  Any non-applicable
 *    rules discard (-ax), and write the new rules subset to 'myrules.conf'
 *    (-out myrules.conf)
 *
 * Effect:
 *
 *    All rules which do not apply to the local network are either discarded
 *    or marked as non-applicable. 
 *
 * Notes:
 *    - The 'any' rules set from rapidnet will create problems for this app, because it
 *      cannot distinguish which is outside and inside traffic.
 *    - There should be no crazy system-specific stuff.  Be warned though, that I have
 *      only tried this under RedHat 6.2
 *
 * Common Error:
 *    - broken pipe : you probably gave nmap bad parameters 
 * 
 * Change Log:
 *
 *   2000.VI.13  : First release (v0.5)
 * 
 * TODO: 
 *     - Make the command-line argument error-checking more user friendly
 *     - Add documentation to the code
 *     - Parse the addresses more effectively 
 *          - Deal with negation operator (i.e. ! ) on addresses
 *          - Add support for more than just one internal address 
 *     - don't assume lower case for ACTION_FLAG (i.e. msg: )
 *     - Support variables
 *     - Support includes of external files 
 *
 */

/* #define DEBUG 1 */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <netdb.h>

#define NMAP_HEADER_2_2x "Port       State       Service"
#define NMAP_HEADER_2_1x "Port    State       Protocol  Service"
#define NMAP_VERSION_2_2x 1
#define NMAP_VERSION_2_1x 2
#define PROGRAM_HEADER1 "SnortRules v0.5 by Roman Danyliw <roman@danyliw.com>"
#define PROGRAM_HEADER2 "== Generates Network Specific Snort Rules using NMap ==" 
#define USAGE  "nmap [parameters] | snortrules -a[x|f] -net[v] <variable> -in <filename> -out <filename>\n"
#define SYNTAX "   -a[x|f]            : action to take for non-applicable rule\n" \
               "                         -ax : discard the rule\n" \
               "                         -af : keep the rule, but flag the alert message\n" \
               "   -net[v] <variable> : name denoting the internal networki (e.g. any, 127.0.0.1/24)\n" \
               "                         -netv : name is a variable (e.g. INTERNAL, _not_ $INTERNAL)\n" \
               "   -in <filename>     : unparsed Snort rules file\n" \
               "   -out <filename>    : output file to be created with parsed rules\n" \
               "   -help              : displays this help and exits\n"
#define EXAMPLE "EXAMPLE:\n" \
                "nmap -sT -sU localhost | snortrules -ax -net INTERNAL -in vision.conf -out myrules.conf\n" \
                "\nRun nmap on the localhost searching for open TCP and UDP ports (-sT -SU). \n" \
                "Read rules from 'vision.conf' (-in vision.conf), using 'INTERNAL' as the \n" \
                "variable representing the internal network (-net INTERNAL).  Any non-applicable\n" \
                "rules discard (-ax), and write the new rules subset to 'myrules.conf'\n" \
                "(-out myrules.conf)\n\n"
#define ACTION_DISCARD 1
#define ACTION_FLAG    2
#define MSG_FLAG "NON-APPLICABLE RULE = "

char open_tcp[65536], open_udp[65536];
char *main_rules_filename, /* = "vision.conf", */
     *main_postprocessed_filename,
     net_variable[20];
char current_rule[1024];
int action;

void ParseArgs(int, char *argv[]);
void FindOpenPorts();
void ProcessRules();
int  isApplicable(char *, char *, char *, char *, char *, char *);
void ConvertPort(char *, int *, int *, char *, int *);
int ResolvePort(char *, char *);
int isOpen(int, int, int, char *);
void FatalError(const char *format, ...);

main (int argc, char *argv[])
{
     int i, service_cnt = 0, udp_cnt = 0, tcp_cnt;
     struct servent *lookup;

     memset(open_tcp, 65536, sizeof(char)); 
     memset(open_udp, 65536, sizeof(char));

     printf("\n%s\n%s\n\n", PROGRAM_HEADER1, PROGRAM_HEADER2);

     ParseArgs(argc, argv);
     FindOpenPorts();

     printf("o Detecting Services\n");
     for (i = 0; i < 65535; i++)
     {
          if (open_tcp[i]) 
          {
               ++service_cnt;
               ++tcp_cnt;
               printf("    %5d/tcp -- ", i);
               lookup = getservbyport(ntohs(i), "tcp");
               if ( lookup != NULL)
                   printf("%s\n", lookup->s_name);
               else
                   printf("???\n");
          }
          if (open_udp[i])
          {
               ++service_cnt;
               ++udp_cnt;
               printf("    %5d/udp -- ", i);
               lookup = getservbyport(ntohs(i), "udp");
               if ( lookup != NULL)
                   printf("%s\n", lookup->s_name);
               else
                   printf("???\n");
          }
     }
     
     if ( service_cnt == 0 )
          printf("\n    Warning: No services were found. A bit odd.\n" \
                 "   Perhaps NMap was not run correctly " \
                 "(try, running with -help)? or\n" \
                 "    An unsupported version of NMap was used.\n" \
                 "    Only versions " \
                 "2.1x and 2.2x were tested (and therefore supported)\n");
     else
     {    
          if ( tcp_cnt == 0 )
               printf("\n    Warning: No TCP services were found. A bit odd.\n" \
                      "    Perhaps [-sT | -sS | -sX | -sN] need to be used with nmap?\n");
          if ( udp_cnt == 0)
               printf("\n    Warning: No UDP services were found. A bit odd.\n" \
                      "    Perhaps [-sU] needs to be used with nmap?\n");
     }

     printf("\no Using %d services to parse rules\n", service_cnt);
     ProcessRules();
}

void ParseArgs(int argc, char *argv[])
{
     int i = 1;

     main_rules_filename = main_postprocessed_filename = NULL;
     memset(&net_variable, 0, sizeof(net_variable));
     action = -1;

     if ( argc == 8 || argc == 2 )
     { 
          if ( argc == 2 && !strcasecmp(argv[1], "-help"))
               FatalError("USAGE : %s%s\n%s", USAGE, SYNTAX, EXAMPLE);
          else if ( argc != 8 )
               FatalError("ERROR: Wrong number of parameters\n\nUSAGE : %s%s\n%s",
                          USAGE, SYNTAX, EXAMPLE);

          while ( i < 8 )
          {
               if ( !strcasecmp(argv[i], "-in") && main_rules_filename == NULL)
               {
                    main_rules_filename = argv[i+1];
                    printf("o Using source file = %s\n", main_rules_filename);
               }
               else if ( !strcasecmp(argv[i], "-out") && main_postprocessed_filename == NULL)
               {
                    main_postprocessed_filename = argv[i+1];
                    printf("o Using destination file = %s\n", main_postprocessed_filename);
               }
               else if ( !strcasecmp(argv[i], "-net") && net_variable[0] == 0)
               {
                    strncpy(net_variable, argv[i+1], 20);
                    printf("o Assuming that internal network = %s\n", net_variable);
               }
               else if ( !strcasecmp(argv[i], "-netv") && net_variable[0] == 0)
               {
                   net_variable[0] = '$';
                   strncat(net_variable, argv[i+1], sizeof(net_variable) - 2);
                    printf("o Assuming that internal network = %s\n", net_variable);
               }
               else if ( !strcasecmp(argv[i], "-ax") && action == -1)
               {
                    action = ACTION_DISCARD;
                    --i;
                    printf("o Discarding Rules when not applicable\n");
               }
               else if ( !strcasecmp(argv[i], "-af") && action == -1)
               {
                    action = ACTION_FLAG;
                    --i;
                    printf("o Flagging Rules when not applicable\n");
               }
               else
               {
                    FatalError("ERROR: Illegal parameter #%d '%s'\n\nUSAGE : %s %s%s\n%s",
                               i, argv[i], argv[0], USAGE, SYNTAX, EXAMPLE);
               }
               i += 2; 
          }
     }
     else
          FatalError("ERROR: Wrong number of parameters\n\nUSAGE : %s %s%s\n%s",
                     argv[0], USAGE, SYNTAX, EXAMPLE);
}

void FindOpenPorts()
{
     char buffer[1024], *port_proto, *port, *proto, *state;
     int nmap_version;

     while ( (fgets(buffer, 1024, stdin) ) != NULL)
     {
           nmap_version = 0;
           if ( strstr(buffer, NMAP_HEADER_2_2x) != NULL )
                nmap_version = NMAP_VERSION_2_2x;
           else if (strstr(buffer, NMAP_HEADER_2_1x) != NULL)
                nmap_version = NMAP_VERSION_2_1x;

           if (nmap_version)
           {
                fgets(buffer, 1024, stdin);
                while(buffer != NULL && buffer[0] != '\n' && buffer[0] != ' ')
                {
                     if (nmap_version == NMAP_VERSION_2_2x)
                     {
                          port_proto = strtok (buffer, " ");
                          port = strtok (port_proto, "/");
                          proto = strtok(NULL, " ");
                     }
                     else if (nmap_version == NMAP_VERSION_2_1x)
                     {
                          port = strtok (buffer, " ");
                          state = strtok (NULL, " ");
                          proto = strtok (NULL, " ");
                     }
                     if ( !strcasecmp(proto, "tcp") )       /* tcp */
                          ++open_tcp[atoi(port)];
                     else if ( !strcasecmp(proto, "udp") )  /* udp */
                          ++open_udp[atoi(port)];
                     fgets(buffer, 1024, stdin);
                } 
           }
     }
}


void ProcessRules()
{
  /* 
     alert <protocol> <source network> <source port> -> <target network> <target port> <signature>
   */
     FILE *rules, *out;
     time_t current_time;
     char buffer[1024], *line, *direction,
          *identifier, *protocol, 
          *source_network, *source_port, *dest_network, *dest_port,
          *pre, *dummy, *msg, *post, *newmsg;
     int total_cnt = 0, applicable_cnt = 0;

     if ( (rules = fopen(main_rules_filename, "r")) == NULL )
       FatalError("ERROR: Could not open the rules file: '%s'\n", main_rules_filename);

     if ( (out = fopen(main_postprocessed_filename, "w")) == NULL )
       FatalError("ERROR: Could not write processed rules to : '%s'\n", main_postprocessed_filename);

     /* Write the program header into the output file */
     time(&current_time);
     fprintf(out, "##### parsed by %s\n" \
                  "##### on %s" \
                  "#################################################################\n", 
                  PROGRAM_HEADER1, ctime(&current_time));
     
     while ( (fgets(buffer, 1024, rules)) != NULL )
     {
          strcpy(current_rule, buffer);
          line = buffer;
          while (line[0] == ' ')  ++line;
  
	  if ( line[0] == '#' || line[0] == '\n')   /* a comment or blank */   
              fprintf(out, "%s", current_rule);
          else
	  {
	       identifier = strtok(line, " ");
               if ( strcasecmp(identifier, "alert") ) 
               /* log, pass, var, preprocessors */
	           fprintf(out, "%s", current_rule);
               else                 /* an alert */
	       {
                 ++total_cnt;     /* increment number of processed rules */
		 protocol = strtok(NULL, " ");        /* protocol name */
                 source_network = strtok(NULL, " ");  /* source network */
                 source_port = strtok(NULL, " ");     /* source port */
                 direction = strtok(NULL, " ");       /* ->, <-, <> */
                 dest_network = strtok(NULL, " ");    /* destination network */
                 dest_port = strtok(NULL, " ");       /* destination port */

                 if ( isApplicable(source_network, source_port,
                                    dest_network, dest_port, protocol, direction) )
                 {
		      ++applicable_cnt;
                      fprintf(out, "%s", current_rule);
                 }
                 else
                 {
#ifdef DEBUG
                      printf("REJECT: %s", current_rule);
#endif
                      if ( action == ACTION_FLAG )     /* don't reject only mark the message */
                      {
                           newmsg = (char *) malloc(strlen(current_rule) + strlen(MSG_FLAG));
                           memset(newmsg, 0, sizeof(newmsg));
                           pre = strtok(current_rule, "msg:"); 
                           dummy = strtok(NULL, "\"");
                           msg = strtok(NULL, "\";");
                           post = strtok(NULL, "");

                           sprintf(newmsg, "%sm%s\"%s%s\"%s", pre, dummy, MSG_FLAG, msg, post);
                           fprintf(out, "%s", newmsg);
                           free(newmsg);
                      }
                 }
	       }      
	  }
     }
     printf("\no Processed %d rules from '%s'\n", total_cnt, main_rules_filename);
     if ( action == ACTION_DISCARD )
          printf("\no %d (of %d) applicable rules were written to '%s'\n", 
                 applicable_cnt, total_cnt, main_postprocessed_filename);
     else if ( action == ACTION_FLAG)
          printf("\no %d (of %d) applicable rules and %d (of %d) non-applicable rules were written to '%s'\n",
                 applicable_cnt, total_cnt, total_cnt-applicable_cnt, total_cnt, main_postprocessed_filename);         
     fclose(rules);
     fclose(out);     
}

int  isApplicable(char *left_addr, char *left_port, char *right_addr, 
                   char *right_port, char *proto, char *direction)
/* This function determines based on the from/to port and address whether
   a given rule is applicable. 
   
   address := <any> | <$INTERNAL> | <$EXTERNAL> | <num><.><num><.><num><.><num></><num>
   address_with_negation := <address> | <!><address>

   Essentially, we only want the rules where the destination address is an open port
   <source addr> <source port> -> $INTERNAL <port>
   $INTERNAL <port>            <- <source addr> <source port>
   $INTERNAL <port>            <> <addr> <port>
   <addr> <port>               <> $INTERNAL <port> 
*/
{
     unsigned int lport_start, lport_end, rport_start, rport_end, rnegate, lnegate;

#ifdef DEBUG2
     printf("left: [%s] right: [%s]\n", left_port, right_port);
#endif

     /* NMap wil only detect open TCP and UDP ports.  No statement about
        ICMP traffic (the only other protocol supported) can be made.  
        So let it pass. */
     if ( strcasecmp(proto, "tcp") && strcasecmp(proto, "udp") )
          return 1;
   
     ConvertPort(left_port, &lport_start, &lport_end, proto, &lnegate);
     ConvertPort(right_port, &rport_start, &rport_end, proto, &rnegate);

#ifdef DEBUG2
     printf("left: [%d - %d] %s right: [%d, %d]\n", lport_start, lport_end, direction,
            rport_start, rport_end); 
#endif

    if ( !strcasecmp(direction, "->"))
         if ( !strcasecmp(right_addr, net_variable) )
         /* <source addr> <source port> -> $INTERNAL <port> */
              return isOpen(rport_start, rport_end, rnegate, proto);
    else if ( !strcasecmp(direction, "<-") )
         if ( !strcasecmp(left_addr, net_variable) )
         /* $INTERNAL <port> <- <source addr> <source port> */
              return isOpen(lport_start, lport_end, lnegate, proto); 
      /* Can't make any assumptions about bi-directional rules */
/*    else if ( !strcasecmp(direction, "<>") )
         if ( !strcasecmp(left_addr, net_variable) ) */
         /* $INTERNAL <port>            <> <addr> <port> */
/*              return isOpen(lport_start, lport_end, lnegate, proto);  
         else if ( !strcasecmp(right_addr, net_variable) ) */
         /* <addr> <port>               <> $INTERNAL <port> */
/*              return isOpen(rport_start, rport_end, rnegate, proto); */
    else  /* got something I don't understand, better include it */
         return 1;

    return 1;     /* never reached */
}

void ConvertPort(char *port, int *port_start, int *port_end, char *proto, int *negate)
/*
   port := <any> | <number> | <number><:><number> | <number><:> | <:><number>
   port_with_negation := <port> | <!><port>
*/
{
     char *t, *strport;

     /* check for negation operator */
     if ( port[0] == '!' )
     {
          strport = &port[1];
          *negate = 1;
     }
     else
     {
          strport = port;
          *negate = 0;
     }

     /* deal with <any> */
     if ( !strcasecmp(strport, "any") )
     {
          *port_start = 0;
          *port_end = 65535;
     }
     /* deal with a <number> */
     else if (isNumber(strport))
     {
          *port_start = atoi(strport);
          *port_end = atoi(strport);
     }
     else if ( strstr(strport, ":") != NULL )
     {
          /* deal with <number><:> */
          if ( strport[strlen(strport)-1] == ':' )
          {
               *port_end = 65535;
               t = (char *) malloc ( strlen(strport) );
               strncpy(t, strport, strlen(strport)-1);
               t[strlen(strport)-1] = 0;  /* not sure why, but got error w/o */
               *port_start = ResolvePort(t, proto);
               free(t); 
          }

          /* deal with <:><number> */
          else if ( strport[0] == ':' )
          {
               *port_start = 0;
               t = &(strport[1]);
               *port_end = ResolvePort(t, proto);
          }

          /* deal with <number><:><number> */ 
          else
          {
               *port_start = ResolvePort(strtok(strport, ":"), proto);
               *port_end = ResolvePort(strtok(NULL, " "), proto); 
          }
     }
     /* deal with a <number> */
     else
     {
          *port_start = ResolvePort(strport, proto);
          *port_end = *port_start;
     }

     if ( *port_start > *port_end )  /* make sure we have a valid port interval */
          FatalError("ERROR: The start port %d > the end port %d in rule %s\n", 
                     *port_start, *port_end, current_rule);

     if ( *port_start < 0 || *port_start > 65535 )
          FatalError("ERROR: The port %d is illegal in rule %s\n",
                     *port_start, current_rule);

     if ( *port_end < 0 || *port_end > 65535 )
          FatalError("ERROR: The port %d is illegal in rule %s\n",
                     *port_end, current_rule);
}

int ResolvePort(char *strport, char *proto)
{
     struct servent *lookup;

     if ( isNumber(strport) )
          return atoi(strport);
     else
     {
          if ( !strcasecmp(proto, "TCP") )
               lookup = getservbyname(strport, "tcp");
          else if ( !strcasecmp(proto, "UDP") )
               lookup = getservbyname(strport, "udp");

          if ( lookup != NULL)
          {
              return ntohs(lookup->s_port);
          }
          else
              FatalError("ERROR: Could not resolve service: '%s' to a port number" \
                         " when processing rule: %s\n", strport, current_rule);
     }
     return 0;
}

int isOpen(int low, int high, int negate, char *proto)
{
     int i;

     if (negate)
     {
          for (i = 0; i < low; i++)
               if (!strcasecmp(proto, "tcp"))
               {
                    if (open_tcp[i])
                         return 1;
               }
               else if (!strcasecmp(proto, "udp"))
               {
                    if (open_udp[i])
                         return 1;
               }
          for (i = high + 1; i <= 65535; i++)
               if (!strcasecmp(proto, "tcp"))
               {
                    if (open_tcp[i])
                         return 1;
               }
               else if (!strcasecmp(proto, "udp"))
               {
                    if (open_udp[i])
                         return 1;
               }
     }
     else
          for (i = low; i <= high; i++)
               if (!strcasecmp(proto, "tcp"))
               {
                    if (open_tcp[i])
                         return 1;
               }
               else if (!strcasecmp(proto, "udp"))
               {
                    if (open_udp[i])
                         return 1;
               }
     return 0;
}

void FatalError(const char *format, ...)
{
     char buf[1024];
     va_list ap;
 
     va_start(ap, format);

     vfprintf(stderr, format, ap);

     exit(-1);
}

int isNumber(char *str)
{
     int i;

     for(i=0; i < strlen(str); i++)
         if ( !isdigit(str[i] ) )
              return 0;
     return 1;
}


