//
//  GameScene.swift
//  FlappyBird
//
//  Created by Michael Gazzola on 2/23/15.
//  Copyright (c) 2015 Michael Gazzola. All rights reserved.
//

import SpriteKit

class GameScene: SKScene, SKPhysicsContactDelegate {
    let numSkyNodes: Int = 5
    let skyHeight: CGFloat = 65.0
    
    let numLandNodes: Int = 5
    let landHeight: CGFloat = -15
    
    let birdCategory: UInt32 = 1 << 0
    let worldCategory: UInt32 = 1 << 1
    let pipeCategory: UInt32 = 1 << 2
    
    override func didMoveToView(view: SKView) {
        self.backgroundColor = SKColor(red: 81.0/255.0, green: 192.0/255.0, blue: 201.0/255.0, alpha: 1)
        self.physicsWorld.gravity = CGVectorMake(0, -10)
        self.physicsWorld.contactDelegate = self
        
        self.setupGame()
    }
    
    func setupGame() {
        
        self.paused = true
        
        // Sky setup.
        for i in 0..<self.numSkyNodes {
            var sky = SKSpriteNode(imageNamed: "sky")
            sky.position = CGPointMake(CGFloat(i) * sky.size.width,
                self.skyHeight)
            sky.name = "sky"
            self.addChild(sky)
        }
        
        // Land setup.
        for i in 0..<self.numLandNodes {
            var land = SKSpriteNode(imageNamed: "land")
            land.position = CGPointMake(CGFloat(i) * land.size.width,
                self.landHeight)
            land.name = "land"
            land.zPosition = 2
            self.addChild(land)
        }
        
        // Bird setup.
        let birdTexture1 = SKTexture(imageNamed: "bird-01")
        birdTexture1.filteringMode = .Nearest
        let birdTexture2 = SKTexture(imageNamed: "bird-02")
        birdTexture2.filteringMode = .Nearest
        
        let flap = SKAction.animateWithTextures([birdTexture1, birdTexture2], timePerFrame: 0.2)
        let flapForever = SKAction.repeatActionForever(flap)
        
        let bird = SKSpriteNode(texture: birdTexture1)
        bird.name = "birdy"
        bird.position = CGPointMake(0.35 * self.frame.size.width,
            0.6 * self.frame.size.height)
        bird.setScale(2)
        bird.runAction(flapForever)
        bird.physicsBody = SKPhysicsBody(circleOfRadius: bird.size.height / 2.0)
        bird.physicsBody?.dynamic = true
        bird.physicsBody?.allowsRotation = false
        bird.physicsBody?.categoryBitMask = self.birdCategory
        bird.physicsBody?.collisionBitMask = self.worldCategory | self.pipeCategory
        bird.physicsBody?.contactTestBitMask = self.worldCategory | self.pipeCategory
        
        self.addChild(bird)
        
        let ground = SKNode()
        ground.position = CGPointMake(0, skyHeight/2)
        ground.physicsBody = SKPhysicsBody(rectangleOfSize:
            CGSizeMake(self.frame.size.width, 2))
        ground.physicsBody?.dynamic = false
        ground.physicsBody?.categoryBitMask = self.worldCategory
        self.addChild(ground)
        
        let pipes = self.generatePipes()
        pipes.zPosition = 1
        pipes.physicsBody?.categoryBitMask = self.pipeCategory
        self.addChild(pipes)
    }
    
    func didBeginContact(contact: SKPhysicsContact) {
        if ((contact.bodyA.categoryBitMask & (self.worldCategory | self.pipeCategory) != 0) || (contact.bodyB.categoryBitMask & (self.worldCategory | self.pipeCategory) != 0)) {
            self.removeAllChildren()
            self.removeAllActions()
            self.setupGame()
        }
    }
    
    override func touchesBegan(touches: NSSet, withEvent event: UIEvent) {
        if (self.paused) {
            self.paused = false
        }
        
        let bird = self.childNodeWithName("birdy")
        bird?.physicsBody?.velocity = CGVectorMake(0, 0)
        bird?.physicsBody?.applyImpulse(CGVectorMake(0, 40))

    }
   
    override func update(currentTime: CFTimeInterval) {
        self.enumerateChildNodesWithName("sky", usingBlock: { (node, _) -> Void in
            var node = node as SKSpriteNode
            node.position = CGPointMake(node.position.x - 1, node.position.y)
            
            if (node.position.x <= -node.size.width) {
                node.position = CGPointMake(
                        node.position.x + node.size.width * CGFloat(self.numSkyNodes),
                        node.position.y)
            }
        })
        
        self.enumerateChildNodesWithName("land", usingBlock: { (node, _) -> Void in
            var node = node as SKSpriteNode
            node.position = CGPointMake(node.position.x - 1, node.position.y)
            
            if (node.position.x <= -node.size.width) {
                node.position = CGPointMake(
                    node.position.x + node.size.width * CGFloat(self.numLandNodes),
                    node.position.y)
            }
        })
        
        let bird = self.childNodeWithName("birdy")
        bird?.zRotation = bird!.physicsBody!.velocity.dy * 0.0015
    }

    // Generates the pipes in FlappySwift.
    func generatePipes() -> SKNode {
        // The node that represents all of the pipes.
        let pipes = SKNode()
        
        // Load the textures.
        let pipeTextureBottom = SKTexture(imageNamed: "PipeUp")
        let pipeTextureTop = SKTexture(imageNamed: "PipeDown")
        pipeTextureBottom.filteringMode = .Nearest
        pipeTextureTop.filteringMode = .Nearest
        
        // Animation that moves the pipes from right to left.
        let distanceToMove = CGFloat(
            self.frame.size.width + 2.0 * pipeTextureBottom.size().width)
        let movePipes = SKAction.moveByX(
            -distanceToMove, y:0.0,
            duration: NSTimeInterval(0.01 * distanceToMove))
        
        // Animation the removes pipes.
        let removePipes = SKAction.removeFromParent()
        
        // Put them together.
        let movePipesAndRemove = SKAction.sequence([movePipes, removePipes])
        
        // Animation which spawns the pipes at a random height.
        // But always makes the pipes 150 pixels away from each other.
        let spawn = SKAction.runBlock({ () -> Void in
            let pipePair = SKNode()
            pipePair.position = CGPointMake(
                self.frame.size.width + pipeTextureBottom.size().width * 2,
                0)
            let maxHeight = UInt32(self.frame.size.height / 6)
            let bottomHeight = arc4random() % maxHeight + maxHeight
            
            // Set up the bottom pipe.
            let bottomPipe = SKSpriteNode(texture: pipeTextureBottom)
            bottomPipe.setScale(3)
            bottomPipe.position = CGPointMake(0.0, CGFloat(bottomHeight))
            bottomPipe.physicsBody = SKPhysicsBody(
                rectangleOfSize: bottomPipe.size)
            bottomPipe.physicsBody?.dynamic = false
            
            // Set up the top pipe.
            let topPipe = SKSpriteNode(texture: pipeTextureTop)
            topPipe.setScale(3)
            topPipe.position = CGPointMake(
                0.0, CGFloat(bottomHeight) + topPipe.size.height + 150.0)
            topPipe.physicsBody = SKPhysicsBody(rectangleOfSize: topPipe.size)
            topPipe.physicsBody?.dynamic = false
            
            // Add them together into the pipe pair.
            pipePair.addChild(bottomPipe)
            pipePair.addChild(topPipe)
            
            // Animate the pair.
            pipePair.runAction(movePipesAndRemove)
            
            // Add to our pipes.
            pipes.addChild(pipePair)
        })
        
        // Pause.
        let delay = SKAction.waitForDuration(NSTimeInterval(2.0))
        
        // Put the spawn and the delay together.
        let spawnThenDelay = SKAction.sequence([spawn, delay])
        self.runAction(SKAction.repeatActionForever(spawnThenDelay))
        
        return pipes
    }
}
