package edu.cmu.cs211.compression.huffman;
/**
 * Represents a node in the Huffman tree. Instances of this class that are internal
 * nodes have exactly two children. Leaf nodes hold the data.
 * 
 * @param <T>
 *            The type of objects stored at the leafs of the Huffman tree.
 */

// Don't modify this file!  In other words:
// Yes, we know it makes sense for this to be Comparable...but don't
// you want practice writing a Comparator?
public class HuffmanNode<T> {

	private T value;
	private int freq;

	private HuffmanNode<T> l, r;
	private HuffmanNode<T> parent;

	/**
	 * Creates a leaf node without a frequency
	 */
	public HuffmanNode (T value)
	{
		this.value = value;
	}

	/**
	 * Creates a leaf node with a given frequency
	 * 
	 * @throws IllegalArgumentException
	 *             if freq is negative
	 */
	public HuffmanNode (int freq, T value)
	{
		if (freq < 0)
			throw new IllegalArgumentException ("frequence needs to be non-negative");

		this.value = value;
		this.freq = freq;
	}

	/**
	 * Creates an internal node with the given childern.
	 * 
	 * @param l
	 *            The left child
	 * @param r
	 *            The right child
	 * @throws IllegalArgumentException
	 *             If either l or r is null.
	 */
	public HuffmanNode (HuffmanNode<T> l, HuffmanNode<T> r)
	{
		if (l == null || r == null)
			throw new IllegalArgumentException ("Trying to create an internal node with a null child");

		this.l = l;
		this.r = r;

		l.parent = r.parent = this;

		this.freq = l.getFreq () + r.getFreq ();
	}

	/**
	 * Checks if this is a leaf node, a node which stores a value in the Huffman
	 * tree.
	 */
	public boolean isLeaf ()
	{
		assert (getLeft () == null) == (getRight () == null) : "Huffman trees should be full";

		return getLeft () == null;
	}

	/** Checks if this is the root of the Huffman tree. */
	public boolean isRoot ()
	{
		return getParent () == null;
	}

	/** Gets the frequency of this node. */
	public int getFreq ()
	{
		return freq;
	}

	/**
	 * Gets the value of this node, or null if this is an internal node
	 */
	public T getValue ()
	{
		return value;
	}

	/**
	 * Gets the parent of this node, or null if this is the root of the Huffman
	 * tree.
	 */
	public HuffmanNode<T> getParent ()
	{
		return parent;
	}

	/**
	 * Gets the left child of this node, or null if this is a leaf.
	 */
	public HuffmanNode<T> getLeft ()
	{
		return l;
	}

	/**
	 * Gets the right child of this node, or null if this is a leaf.
	 */
	public HuffmanNode<T> getRight ()
	{
		return r;
	}
}
