import java.io.BufferedReader;
import java.io.FileReader;
import java.util.StringTokenizer;

/*
 * This class models a swamp. It can read it from a file, identify
 * the start and finish, as well as the paths, and be viewed as a 
 * String, &c
 */
public class Swamp 
{
    private boolean swamp[][]; // 2D swamp; true=path, false=wall
    private boolean scratch[][];//scratch space

    //how many squares can we jump?
    private int jump;

    // Start position
    private int start_row;
    private int start_col;

    // Models a cell, or (row, col) pair within the swamp. This is basically 
    // here so that we can return both coordinates at once.
    public static class Cell
  {
    // The row and the column of some cell (path or wall) within the swamp 
    private int row;
    private int col;

    // Returns the column in which this cell lives
    public int getCol()
    {
	return col;
    }

    // Returns the row in which this cell lives
    public int getRow()
    {
	return row;
    }

    // "true" if two cells have the same (row, col) pair, "false" otherwise
    public boolean isEqual(Object obj)
    {
	try{
	    Cell c = (Cell)obj;
	    return row==c.row && col==c.col;
	}catch(Exception e){}
	return false;
    }

    // Returns String representation of a Cell.
    public String toString()
    {
      return "(" + row + "," + col + ")";
    }

    // A very simple constructor
    public Cell(int row, int col)
    {
      this.row = row;
      this.col = col;
    }
  }

  // This gets thrown if we detect a corrupt input file
  public class InvalidSwampFileException extends Exception
  {
    private int lineNum; // The current line (what we just read) of input

    // Returns the current line of the input file
    public int getLineNumber()
    {
      return lineNum;
    }

    // String representation of the exception: basically, the line number
    public String toString()
    {
      String s = super.toString();
      s = s + " at line: " + lineNum;
      return s;
    }

    // A very simple constructor
    public InvalidSwampFileException(String message, int lineNumber)
    {
      super(message);
      this.lineNum = lineNumber;
    }
  }


  // This class modesl the swamp: a 2D array containing walls and paths
  public Swamp(String filename) throws InvalidSwampFileException
  {
    int lineNum = 0;
    int numCols = 0;
    int numRows = 0;

    BufferedReader swampFile = null;

    StringTokenizer lineTokenizer = null;
    String token = null;
    String line = null;

    try
    {
      swampFile = new BufferedReader(new FileReader(filename));
      while(lineNum < 4) 
      {
        lineNum++;

        line = swampFile.readLine();
        lineTokenizer = new StringTokenizer(line);

        token = lineTokenizer.nextToken();

        if(token.equalsIgnoreCase("Rows:"))
        {
          numRows = Integer.parseInt(lineTokenizer.nextToken());
        }
        else if(token.equalsIgnoreCase("Cols:"))
        {
          numCols = Integer.parseInt(lineTokenizer.nextToken());
        }
        else if(token.equalsIgnoreCase("Jump:"))
        {
          jump = Integer.parseInt(lineTokenizer.nextToken());
        }
        else if(token.equalsIgnoreCase("Start:"))
        {
          lineTokenizer = new StringTokenizer(line, "(,)");
          lineTokenizer.nextToken();

          start_row = Integer.parseInt(lineTokenizer.nextToken());
          start_col = Integer.parseInt(lineTokenizer.nextToken());
        } 
        else
        {
          throw new InvalidSwampFileException("Invalid Swamp File", lineNum);
        }
      }

      if((numRows <= 0) || (numCols <= 0))
      {
        throw new InvalidSwampFileException("Invalid Swamp File", lineNum);
      }

      scratch = new boolean [numRows][numCols];
      swamp = new boolean [numRows][numCols];
      for(int row = 0; row < numRows; row++)
      {
        for(int col = 0; col < numCols; col++)
        {
	    scratch[row][col] = swamp[row][col] = false;
        }
      }

      //Read in each line of the file
      int row, col;
      lineNum++;

      while((line = swampFile.readLine()) != null)
      {
        //Ignore empty lines
        if(!line.equals(""))
        {
          //Get the row and column out of the line
          lineTokenizer = new StringTokenizer(line, "(,)");
          row = Integer.parseInt(lineTokenizer.nextToken());
          col = Integer.parseInt(lineTokenizer.nextToken());

          swamp[row][col] = true;
        }

        lineNum++;
      }
    }
    catch(Exception e)
    {
      throw new InvalidSwampFileException("Invalid Swamp File", lineNum);
    }
  }

  // Returns the coordinates of the start block
  public Cell getStart()
  {
      return new Cell(start_row, start_col);
  }

  // "true" if the cell is out of bounds
  public boolean isFinish(Cell cell)
  {
      return cell.col <= jump || cell.row <= jump || 
	  cell.col >= swamp.length -1 - jump || cell.row >= swamp.length -1 - jump;
  }

  // "true" if the cell contains is a path 
  // Note: Returns false if out of bounds
  // Note: paths are "true", walls are "false" within the swamp array
  public boolean isPath(Cell cell)
  {
      try{
	  return swamp[cell.row][cell.col];
      }catch(Exception e){}
      return false;
  }

  // Make a String representation of the Swamp. X for wall, space for path,
  // S for start, F for finish.
  // See the assignment doucmentation for sample output
  public String toString()
  {
      String ret="  ";
      for(int i=0;i<swamp.length;i++)ret+=" "+i;
      ret+="\n";
      for(int i=0;i<swamp.length;i++){
	  ret+=i+" ";
	  for(int j=0;j<swamp[i].length;j++){
	      if(i==start_row && j==start_col)ret+=" S";
	      else if(swamp[i][j])ret+=" #";
	      else ret+="  ";
	  }
	  ret+="\n";
      }
      return ret;
  }

    public void markCell(int row, int col){
	try{
	    scratch[row][col] = true;
	}catch(Exception e){}
    }
    public boolean isMarked(int row, int col){
	try{
	    return scratch[row][col];
	}catch(Exception e){
	    return true;
	}
    }
    public void clearMarks(){
	for(int i=0;i<scratch.length;i++)
	    for(int j=0;j<scratch[i].length;j++)
		scratch[i][j] = false;
    }
    public int getJump(){
	return jump;
    }
}
