#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include "bmplib-int.h"
#include "bmplib.h"

/*
 * This method enlarges a 24-bit, uncompressed .bmp file
 * that has been read in using readFile()
 *
 * THIS IS THE METHOD THAT YOU SHOULD WRITE
 *
 * original - an array containing the original pixels, 3 bytes per each
 * rows     - the original number of rows
 * cols     - the original number of columns
 *
 * scale    - the multiplier applied to EACH OF the rows and columns, e.g.
 *           if scale=2, then 2* rows and 2*cols
 *
 * new      - the new array containing the pixels, allocated within
 * newrows  - the new number of rows (scale*rows)
 * newcols  - the new number of cols (scale*cols)
 */
int enlarge (pixel *original, int rows, int cols, int scale, 
             pixel **new, int *newrows, int *newcols) {
  return 0;
}


/*
 * This method rotates a 24-bit, uncompressed .bmp file that has been read 
 * in using readFile(). The rotation is expressed in degrees and can be
 * positive, negative, or 0 -- but it must be a multiple of 90 degrees
 * 
 *
 * original - an array containing the original pixels, 3 bytes per each
 * rows     - the number of rows
 * cols     - the number of columns
 * rotation - a positive or negative rotation, 
 *
 * new      - the new array containing the pixels, allocated within
 */
int rotate (pixel *original, pixel **new, int rotation, int rows, int cols) {
  return 0;
}


/*
 * This method vertically flips a 24-bit, uncompressed .bmp file
 * that has been read in using readFile(). This can be used to correct
 * for the fact that images are initially upside-down
 * 
 * THIS IS GIVEN TO YOU SOLELY TO LOOK AT AS AN EXAMPLE
 * TRY TO UNDERSTAND HOW IT WORKS
 *
 * original - an array containing the original pixels, 3 bytes per each
 * rows     - the number of rows
 * cols     - the number of columns
 *
 * new      - the new array containing the pixels, allocated within
 */
int flip (pixel *original, pixel **new, int rows, int cols) {
  if ((rows <= 0) || (cols <= 0))
    return -1;
  *new = (pixel *) malloc (rows*cols*sizeof(pixel));

  for (rowIndex=0; rowIndex < rows; rowIndex++)
    for (colIndex=0; colIndex < cols; colIndex++)
      *((*new) + ((((rows-1)-rowIndex)*cols) + colIndex)) = 
        *(original + rowIndex*cols + colIndex);

  return 0;
}


/*
 * Reads an uncompressed 24-bit .bmp file from a file into an
 * array of pixels, notes the dimensions
 *
 * filename - the .bmp file's name
 * image    - an array containing the original pixels, 3 bytes per each
 * rows     - the number of rows
 * cols     - the number of columns
 */
int readFile (char *filename, int *rows, int *cols, pixel** image) {
  int fd;
  unsigned int start;

  if ((fd = open (filename, O_RDONLY)) < 0) {
    perror("Problem with input file");
    return -1;
  }

  readHeader (fd, rows, cols, &start);
  *image = (pixel *) malloc (sizeof(pixel) * (*rows) * (*cols));
  readBits (fd, *image, *rows, *cols, start);

  close (fd);

  return 0;

}


/*
 * Writes an uncompressed 24-bit .bmp file, given an array of pixels, 
 * and the dimensions
 *
 * filename - the .bmp file's name
 * image    - an array containing the original pixels, 3 bytes per each
 * rows     - the number of rows
 * cols     - the number of columns
 */
int writeFile (char *filename, int rows, int cols, pixel *image) {
  int fd;
  unsigned int start = DEFAULT_OFFSET;

  if ((fd = open (filename, O_RDWR | O_CREAT | O_TRUNC, 0666)) < 0) {
    perror("Problem with output file");
    return -1;
  }

  writeHeader (fd, rows, cols, start);
  writeBits(fd, rows, cols, image, start);

  close (fd);
  return 0;
}
  


/*
 * Don't worry about this
 */
static int readHeader(int fd, int *rows, int *cols, unsigned int *start) {

  BITMAPFILEHEADER bmfh;
  BITMAPINFOHEADER bmih;

  read (fd, (BITMAPFILEHEADER *)((unsigned long)(&bmfh) + 2), sizeof(bmfh)-2);
  read (fd, &bmih, sizeof(bmih));

  *rows = bmih.biHeight;
  *cols = bmih.biWidth;
  *start = bmfh.bfOffBits;

  return 0;

}


/*
 * Don't worry about this
 */
static int writeHeader(int fd, unsigned int rows, unsigned int cols, 
                       unsigned int start) {

  unsigned int fileSize;
  unsigned int headerSize;

  unsigned int paddedCols;
  
  int count;

  BITMAPFILEHEADER bmfh;
  BITMAPINFOHEADER bmih;

  memset (&bmfh, 0, sizeof(bmfh));
  memset (&bmih, 0, sizeof(bmih));

  paddedCols = ((cols/4)*4 !=cols ? ((cols+4)/4)*4 : cols);

  headerSize = sizeof(BITMAPFILEHEADER) + sizeof(BITMAPINFOHEADER);
  fileSize = rows*paddedCols*sizeof(pixel)+headerSize;

  bmfh.bfType = 0x4D42;
  bmfh.bfSize = fileSize;
  bmfh.bfReserved1 = 0;
  bmfh.bfReserved2 = 0;
  bmfh.bfOffBits = start;
  bmih.biSize = 40;
  bmih.biWidth  = cols;
  bmih.biHeight = rows;
  bmih.biPlanes = 1;
  bmih.biBitCount = 24;
  bmih.biCompression = 0;
  bmih.biSizeImage = 0;
  bmih.biXPelsPerMeter  = 0;
  bmih.biYPelsPerMeter = 0;
  bmih.biClrUsed = 0;
  bmih.biClrImportant = 0;

  write (fd, (BITMAPFILEHEADER *)((unsigned long)(&bmfh) + 2), sizeof(bmfh)-2);
  count = write (fd, &bmih, sizeof(bmih));

  return 0;

}


/*
 * Don't worry about this
 */
static int writeBits(int fd, int rows, int cols, 
                     pixel *image, unsigned int start) {
  int row;

  char padding[3];
  int padAmount;

  padAmount = ((cols * sizeof(pixel)) % 4) ? (4 - ((cols * sizeof(pixel)) % 4)) : 0;

  memset(padding, 0, 3);

  lseek (fd, start, SEEK_SET);
  
  fflush (stdout);

  for (row=0; row < rows; row++) {
    write (fd, image + (row*cols), cols*sizeof(pixel));
    write (fd, padding, padAmount);
  }

  return 0;
}



/*
 * Don't worry about this
 */
static int readBits (int fd, pixel *image, int rows, int cols, 
                     unsigned int start) {

  int row;
  char padding[3];
  int padAmount = 0;

  padAmount = ((cols * sizeof(pixel)) % 4) ? (4 - ((cols * sizeof(pixel)) % 4)) : 0;

  lseek (fd, start, SEEK_SET);

  for (row=0; row < rows; row++) {
    read (fd, image + (row*cols), cols*sizeof(pixel));
    read (fd, padding, padAmount);
  }

  return 0;
}
